package ru.yandex.tours.indexer.hotels

import ru.yandex.tours.app.{AkkaSupport, Application, CommonDataHolders, HttpSupport, _}
import ru.yandex.tours.avatars.AvatarSupport
import ru.yandex.tours.db.{HotelsDbSupport, MySqlSupport}
import ru.yandex.tours.extdata.ExtDataSupport
import ru.yandex.tours.indexer.hotels.feed.{HttpPartnerFeedRetriever, PartnerFeedRetriever}
import ru.yandex.tours.indexer.hotels.parsers._
import ru.yandex.tours.indexer.hotels.parsers.booking.S3Hotel2RegionRetriever
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.image.ImageProviders
import ru.yandex.tours.model.image.ImageProviders.ImageProvider
import ru.yandex.tours.parsers.common.DBStableIds
import ru.yandex.tours.partners.{PartnerConfig, PartnerConfigs}
import ru.yandex.vertis.scheduler.model.Task

import scala.concurrent.duration._

trait PartnerIndexingSupport extends SchedulerTasks {
  app: Application
    with MySqlSupport
    with AvatarSupport
    with CommonDataHolders
    with SchedulerSupport
    with HttpSupport
    with AkkaSupport
    with ExtDataSupport
    with S3Support
    with HotelsDbSupport =>

  private implicit val ec = akkaSystem.dispatchers.lookup("akka.actor.hotels-indexer-dispatcher")

  private val stableIds = new DBStableIds(db)
  private val stableIdsEnricher = new StableIdsEnricher(stableIds)
  private val imageEnricher = new ImageEnricher(avatarClient, shouldDownloadNew = isEnvironmentStable)(ec, akkaSystem)
  private val slopes = Slopes.from(extDataService, extDataUpdateChecker, lazyLoad = true)
  private val slopesEnricher = new SlopeDistanceEnricher(slopes, skiResorts, regionTree)
  private val addInfoEnricher = new AddInfoEnricher(slopesEnricher)(akkaSystem, ec)
  private val dbUpdater = new PartnerHotelsDbUpdater(hotelsDao)
  private val partner2config = PartnerConfigs.parse(config)

  private val hotelIndexers = List(
    buildIndexer(Partners.lt, ImageProviders.lt),
    buildIndexer(Partners.sodis, ImageProviders.sodis),
    buildIndexer(Partners.hotels101, ImageProviders.hotels101),
    buildIndexer(Partners.booking, ImageProviders.booking),
    buildIndexer(Partners.ostrovokv3, ImageProviders.ostrovokv3),
    buildIndexer(Partners.oktogo, ImageProviders.oktogo),
    buildIndexer(Partners.rozahutor, ImageProviders.rozahutor),
    buildIndexer(Partners.travelline, ImageProviders.travelline),
    buildIndexer(Partners.backa, ImageProviders.ya),
    buildIndexer(Partners.inna, ImageProviders.inna),
    buildIndexer(Partners.vashotel, ImageProviders.vashotel),
    buildIndexer(Partners.kurortExpress, ImageProviders.kurortExpress),
    buildIndexer(Partners.hotelsCombined, ImageProviders.hotelsCombined)
  )

  override def schedulingTasks: Seq[Task] = super.schedulingTasks ++ hotelIndexers.map(_.asTask)

  private def buildIndexer(partner: Partner, imageProvider: ImageProvider) = {
    val partnerConfig = partner2config.getOrElse(partner, sys.error(s"Can not find config for $partner"))
    val indexer = new PartnerHotelsIndexer(
      db,
      hotelsDao,
      createFeedRetriever(partnerConfig),
      createParser(partner),
      stableIdsEnricher,
      imageEnricher,
      addInfoEnricher,
      dbUpdater,
      partner,
      imageProvider)
    new PartnerHotelIndexRunner(indexer, partner, 24.hours)
  }

  private def createParser(partner: Partner): PartnerHotelParser = partner match {
    case Partners.lt =>
      new CommonFormatParser(partner, regionParam = Some("resortId"))
    case Partners.sodis =>
      new CommonFormatParser(partner, regionParam = Some("PartnerRegionID"))
    case Partners.hotels101 =>
      new CommonFormatParser(partner, regionParam = Some("PartnerRegionID"))
    case Partners.oktogo =>
      new OktogoFormatParser()
    case Partners.booking =>
      new BookingZipFormatParser(regionTree, iso2country, new S3Hotel2RegionRetriever(s3Client))
    case Partners.ostrovokv3 =>
      new OstrovokV3FormatParser()
    case Partners.rozahutor =>
      new CommonFormatParser(partner)
    case Partners.travelline =>
      new CommonFormatParser(partner)
    case Partners.backa =>
      new BackaFormatParser(regionTree)
    case Partners.inna =>
      new CommonFormatParser(partner, regionParam = Some("PartnerRegionID"))
    case Partners.vashotel =>
      new CommonFormatParser(partner, regionParam = Some("PartnerRegionID"))
    case Partners.kurortExpress =>
      new CommonFormatParser(partner, regionParam = Some("PartnerRegionID"))
    case Partners.hotelsCombined =>
      new CommonFormatParser(partner, regionParam = Some("PartnerRegionID"))
    case _ => sys.error(s"Can not create parser due to unknown partner: $partner")
  }

  def createFeedRetriever(partnerConfig: PartnerConfig): PartnerFeedRetriever = {
    new HttpPartnerFeedRetriever(
      client = if (partnerConfig.isExternalUrl) externalHttpClient else httpClient,
      uri = partnerConfig.feedUrl,
      partner = partnerConfig.partner,
      gunzip = partnerConfig.gunzipFeed,
      headers = partnerConfig.headers
    )
  }
}
