package ru.yandex.tours.indexer.hotels

import java.io.InputStream

import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.extdata.{DataTypes, DataDef}
import ru.yandex.tours.geo
import ru.yandex.tours.geo.GeoIndex
import ru.yandex.tours.indexer.hotels.Slopes.Slope
import ru.yandex.tours.model.MapRectangle
import ru.yandex.tours.model.geo.MapObject
import ru.yandex.tours.util.Collections._
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.parsing.{IntValue, DoubleValue, Tabbed}

class Slopes(slopes: Iterable[Slope]) extends GeoIndex(slopes) {

  /**
   *
   * @param lat - latitude
   * @param lon - longitude
   * @return Some(distance) in meters if its less than 15.000 m, None otherwise
   */
  def getSlopeDistance(lat: Double, lon: Double): Option[Int] = {
    val candidates = this.inRectangle(MapRectangle.byCenterAndSpan(lon, lat, 1, 1))
    candidates.map(minDistance(lat, lon, _)).minOpt.map(_ * 1000).map(_.toInt).filter(_ < 15000)
  }

  private def minDistance(lat: Double, lon: Double, slope: Slope) = {
    slope.points.sliding(2).map {
      case Seq(point1, point2) => geo.distanceToSegment(point1.lat, point1.lon, point2.lat, point2.lon, lat, lon)
    }.min
  }
}

object Slopes extends DataDef[Slopes] {

  case class Point(lat: Double, lon: Double)

  case class Slope(id: Int, points: Seq[Point]) extends MapObject {
    require(points.size > 1, "Slope should contains at least 2 points")

    override def longitude: Double = points.head.lon

    override def latitude: Double = points.head.lat
  }

  override def dataType: DataType = DataTypes.slopes

  override def parse(is: InputStream): Slopes = {
    val slopes = IO.readLines(is).map {
      case Tabbed(DoubleValue(lat), DoubleValue(lon), IntValue(id), IntValue(index)) => id -> (lat, lon, index)
    }.toSeq.groupBy(_._1).collect {
      case (id, seq) if seq.size > 1 => Slope(id, seq.map(_._2).sortBy(_._3).map(x => Point(x._1, x._2)))
    }
    new Slopes(slopes)
  }

  def empty = new Slopes(Iterable.empty[Slope])
}
