package ru.yandex.tours.indexer.hotels

import java.io.File

import ru.yandex.tours.model.hotels.HotelsHolder.{PartnerHotel, RawPartnerHotel}
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.parsers.common.StableIds
import ru.yandex.tours.util.lang._
import ru.yandex.tours.util.{IO, Logging, ProtoIO}

import scala.concurrent.{ExecutionContext, Future}
import scala.util.Try

class StableIdsEnricher(stableIds: StableIds)(implicit ec: ExecutionContext) extends Logging {
  /**
   *
   * @param file with delimited [[RawPartnerHotel]]
   * @param partner which all hotels belongs to
   * @return file with delimited [[PartnerHotel]] with set stable ids
   */
  def enrich(file: File, partner: Partner): Future[File] = {
    for {
      allIds <- getIds(file, partner).toFuture
      oldIds <- stableIds.getIds(partner)
      _ <- insertUnknown(partner, oldIds.map(_._1), allIds)
      ids <- stableIds.getIds(partner)
      file <- writeHotelsWithStableIds(file, ids.toMap).toFuture
    } yield file
  }

  private def writeHotelsWithStableIds(file: File, partnerId2id: Map[String, Int]) = Try {
    var withoutId = 0
    IO.usingTmp("stable_ids_enricher") { os =>
      hotels(file).foreach { hotel =>
        partnerId2id.get(hotel.getPartnerId) match {
          case Some(id) =>
            PartnerHotel.newBuilder()
              .setId(id)
              .setRawHotel(hotel)
              .build()
              .writeDelimitedTo(os)
          case None =>
            withoutId += 1
        }
      }
      if (withoutId != 0) log.warn(s"$withoutId hotels without id! Very-very strange situation")
    }
  }

  private def getIds(file: File, partner: Partner) = Try {
    hotels(file).map { hotel =>
      hotel.ensuring(_.getPartner == partner.id, "Different partners!")
      hotel.getPartnerId
    }
  }

  private def insertUnknown(partner: Partner,
                            oldIds: Iterable[String],
                            allIds: Iterator[String]): Future[Unit] = {
    val knownIds = oldIds.toSet
    val idsToAdd = allIds.filter(!knownIds.contains(_)).toVector
    log.info(s"${idsToAdd.size} new stable ids added")
    stableIds.save(partner, idsToAdd.toIterable)
  }

  private def hotels(file: File) = ProtoIO.loadFromFile(file, RawPartnerHotel.PARSER)
}
