package ru.yandex.tours.indexer.hotels.feed

import java.io.{File, FileInputStream}
import java.util.zip.GZIPInputStream

import akka.util.Timeout
import org.apache.commons.io.IOUtils
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.{IO, Logging, Statistics}
import spray.http.{StatusCodes, Uri}

import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, Future}

trait PartnerFeedRetriever {
  def retrieve: Future[File]
}

class HttpPartnerFeedRetriever(client: AsyncHttpClient,
                               uri: Uri,
                               partner: Partner,
                               gunzip: Boolean = false,
                               headers: List[(String, String)] = Nil)
                              (implicit ec: ExecutionContext) extends PartnerFeedRetriever with Logging {
  override def retrieve: Future[File] = {
    val tmpFile = IO.newTempFile(s"${partner}_feed", "tmp")
    val result = client.downloadToFile(uri, tmpFile, headers)(Timeout(60.minutes)).map {
      case StatusCodes.OK => tmpFile
      case sc => sys.error(s"$sc response to request $partner feed")
    }.map { feed =>
      if (gunzip) {
        Statistics.logTime(s"Gunzip $partner feed") {
          IO.using(new GZIPInputStream(new FileInputStream(feed))) { is =>
            IO.usingTmp("partner_feed_retriever") { os =>
              IOUtils.copy(is, os)
              IO.deleteFile(feed)
            }
          }
        }
      } else feed
    }
    result.onFailure {
      case _ => IO.deleteFile(tmpFile)
    }
    result
  }
}

class StaticPartnerFeedRetriever(file: File) extends PartnerFeedRetriever {
  override def retrieve: Future[File] = Future.successful {
    IO.usingTmp("static_feed_retriever") { os =>
      IO.using(new FileInputStream(file)) { is =>
        IOUtils.copy(is, os)
      }
    }
  }
}
