package ru.yandex.tours.indexer.hotels.parsers

import java.io.{File, InputStream, OutputStream}
import java.util.zip.GZIPInputStream

import ru.yandex.common.xml.{XParser, XTag}
import ru.yandex.tours.geo.base.region.{Tree, Types}
import ru.yandex.tours.indexer.hotels.PartnerHotelParser
import ru.yandex.tours.model.BaseModel.LangToVal
import ru.yandex.tours.model.Languages
import ru.yandex.tours.model.hotels.HotelsHolder.{Address, RawPartnerHotel}
import ru.yandex.tours.model.hotels.{Star, Features, Partners}
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.util.lang._
import ru.yandex.tours.util.{Tar, IO, Logging}

import scala.collection.JavaConverters._
import scala.concurrent.Future
import scala.util.{Failure, Success, Try}

class BackaFormatParser(tree: Tree) extends PartnerHotelParser with Logging {
  /**
   * @param is - input stream to parse
   * @return file, which contains delimited [[RawPartnerHotel]]
   */
  override def parse(is: InputStream): Future[File] = Try {
    IO.usingTmp("backa_format_parser") { os =>
      Tar.readFile(is, "companies2.xml.gz") { fileStream =>
        parse(new GZIPInputStream(fileStream), os)
      }
    }
  }.toFuture

  private val hotelRubrics = Set("184106414", "184106404", "184106400", "20699506347")

  def parse(is: InputStream, os: OutputStream): Unit = {
    var failed = 0
    var success = 0
    var skipped = 0
    var noGeoId = 0
    new XParser {
      override def build(): XTag = new XTag() {
        def language = getAttr("xml:lang").getOrElse("ru")

        def isHidden = getAttr("hide").contains("1")

        tag("company") = handle {
          var rubrics = Set.empty[String]
          val backaId = attr("id")
          val builder = RawPartnerHotel.newBuilder()
            .setPartnerId(backaId)
            .setPartnerUrl(s"https://maps.yandex.ru/org/$backaId")
          var address = Map.empty[String, String]

          complete {
            if ((rubrics & hotelRubrics).nonEmpty) {
              if (!builder.hasRegionId) {
                noGeoId += 1
              } else {
                val geoId = builder.getRegionId.toInt
                builder.addAllAddress(createAddress(geoId, address).asJava).setPartner(partner.id)
                Try(builder.build()) match {
                  case Success(rawHotel) => success += 1; rawHotel.writeDelimitedTo(os)
                  case Failure(e) =>
                    log.debug("Can not parse", e)
                    failed += 1;
                }
              }
            } else {
              skipped += 1
            }
          }

          tag("name") = handleComplete {
            val name = LangToVal.newBuilder().setLang(language).setValue(text).build()
            getAttr("xml:lang") match {
              case Some(_) => builder.addName(name)
              case _ => builder.addSynonyms(name)
            }
          }
          tag("geo") = handle {
            tag("address") = handle {
              tag("formatted") = handleComplete {
                val lang = language
                if (!address.contains(lang)) {
                  address += lang -> text
                }
              }
            }
            tag("location") = handle {
              tag("geoid") = handleComplete {
                builder.setRegionId(text)
              }
              tag("gml:pos") = handleComplete {
                Try {
                  val coords = text.trim.split(" ").map(_.toDouble)
                  builder.getPointBuilder.setLatitude(coords(1)).setLongitude(coords(0))
                }
              }
            }
          }
          tag("rubrics") = handle {
            tag("rubric") = handleComplete {
              rubrics += attr("ref")
            }
          }

          tag("url") = handleComplete {
            if (!isHidden) {
              builder.setHotelUrl(text)
            }
          }
          tag("phones") = handle {
            tag("phone") = handle {
              if (!isHidden) {
                tag("formatted") = handleComplete {
                  builder.addPhone(text)
                }
              }
            }
          }
          tag("link") = handle {
            if (getAttr("type").contains("attribution")) {
              val aref = attr("aref")
              tag("atom:link") = handleComplete {
                builder.addAddInfoBuilder().setValue(attr("href")).setName(aref)
              }
            }
          }
          tag("features") = handle {
            tag("feature") = handle {
              val name = attr("id")
              tag("value") = handleComplete {
                val value = text match {
                  case "0" => "false"
                  case "1" => "true"
                  case x => x
                }
                if (name == "star") {
                  val star = Star.getStar(value).id
                  builder.setStars(star)
                } else {
                  if (Features.parse(name, value).nonEmpty) {
                    builder.addFeaturesBuilder().setName(name).setValue(value)
                  }
                }
              }
            }
          }
        }
      }
    }.setLowerCase(true).parse(is)
    log.info(s"Backa hotels parsing finished. " +
      s"Failed: $failed, success $success, skipped: $skipped, without geo id: $noGeoId")
  }

  protected def createAddress(geoId: Int, fullAddresses: Map[String, String]): Iterable[Address] = {
    val pathToRoot = tree.pathToRoot(geoId)
    fullAddresses.map {
      case (stringLang, fullAddress) =>
        val lang = Try(Languages.withName(stringLang)).getOrElse(Languages.ru)
        val address = Address.newBuilder().setLang(lang.toString).setFullAddress(fullAddress)
        tree.country(geoId).map(_.name(lang)).foreach(address.setCountry)
        pathToRoot.find(r => r.`type` == Types.City || r.`type` == Types.Village).map(_.name(lang)).foreach(address.setLocality)
        address.build()
    }
  }

  override def partner: Partner = Partners.backa
}
