package ru.yandex.tours.indexer.hotels.parsers

import java.io.{File, FileInputStream, InputStream, OutputStream}
import java.util.zip.{ZipEntry, ZipInputStream}

import org.apache.commons.io.IOUtils
import ru.yandex.tours.geo.Iso2Country
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.indexer.hotels.PartnerHotelParser
import ru.yandex.tours.indexer.hotels.parsers.booking.{BookingHotel2RegionRetriever, StaticHotel2RegionRetriever}
import ru.yandex.tours.model.hotels.HotelsHolder.RawPartnerHotel
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.util.{IO, ProtoIO, Statistics}

import scala.concurrent.{ExecutionContext, Future}

class BookingZipFormatParser(tree: Tree, iso2country: Iso2Country, regionRetriever: BookingHotel2RegionRetriever)
                            (implicit ec: ExecutionContext) extends PartnerHotelParser {
  /**
   * @param is - input stream to zipped booking feed to parse
   * @return file, which contains delimited [[RawPartnerHotel]]
   */
  override def parse(is: InputStream): Future[File] = {
    val retrieve = Statistics.asyncLogTime("Booking hotel 2 region map retrieve", regionRetriever.retrieve)
    retrieve.flatMap { hotel2region =>
      val parser = createParser(hotel2region)
      IO.usingAsyncTmp("booking_zip_format_parser") { os =>
        val zipIs = new ZipInputStream(is)
        val files = Iterator.continually(zipIs.getNextEntry).takeWhile(_ != null).map(copyZipEntry(_, zipIs))
        parseToSingleFile(files, os, parser)
      }
    }
  }

  private def parseToSingleFile(filesToParse: Iterator[File],
                                os: OutputStream,
                                parser: SingleFileBookingFormatParser): Future[Unit] = {
    if (!filesToParse.hasNext) {
      Future.successful {}
    } else {
      val file = filesToParse.next()
      parser.parse(file).map { file ⇒
        try copyHotels(file, os)
        finally IO.deleteFile(file)
      }.andThen {
        case _ => IO.deleteFile(file)
      }.flatMap(_ => parseToSingleFile(filesToParse, os, parser))
    }
  }

  private def copyZipEntry(ze: ZipEntry, zipIs: ZipInputStream): File = {
    IO.usingTmp("booking_parser_copy") { feedOs =>
      IOUtils.copy(zipIs, feedOs)
    }
  }

  private def copyHotels(input: File, os: OutputStream) = {
    IO.using(new FileInputStream(input)) { is =>
      ProtoIO.loadFromFile(input, RawPartnerHotel.PARSER).foreach(_.writeDelimitedTo(os))
    }
  }

  private def createParser(hotel2region: Map[String, String]) = {
    new SingleFileBookingFormatParser(tree, iso2country, new StaticHotel2RegionRetriever(hotel2region))
  }

  override def partner: Partner = Partners.booking
}
