package ru.yandex.tours.indexer.hotels.parsers

import java.io.{File, InputStream, OutputStream}

import ru.yandex.common.xml.{XParser, XTag}
import ru.yandex.tours.indexer.hotels.PartnerHotelParser
import ru.yandex.tours.model.Languages
import ru.yandex.tours.model.Languages.Lang
import ru.yandex.tours.model.hotels.HotelsHolder.{Address, Feature, RawPartnerHotel}
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.hotels.Star
import ru.yandex.tours.util.{IO, Logging}
import ru.yandex.tours.util.lang._

import scala.collection.JavaConversions._
import scala.concurrent.Future
import scala.util.control.NonFatal
import scala.util.{Failure, Success, Try}

class CommonFormatParser(val partner: Partner,
                         regionParam: Option[String] = None,
                         defaultLang: Lang = Languages.en,
                         featureMappings: PartialFunction[(String, String), (String, String)] = PartialFunction.empty)
  extends PartnerHotelParser with Logging {

  /**
   * @param is - input stream to parse
   * @return file, which contains delimited [[RawPartnerHotel]]
   */
  override def parse(is: InputStream): Future[File] = Try {
    IO.usingTmp("common_format_parser") { os =>
      createParser(is, os)
    }
  }.toFuture

  def createParser(is: InputStream, os: OutputStream): Unit = {
    var failed = 0
    var success = 0
    var total = 0
    new XParser {
      override def build(): XTag = new XTag() {
        def language: Option[String] = getAttr("lang") match {
          case Some(l) if Languages.values.exists(_.toString == l) => Some(l)
          case Some(l) => None                    // unknown language => ignore
          case None => Some(defaultLang.toString) // missing attribute => using default
        }

        tag("company") = handle {
          total += 1
          val builder = RawPartnerHotel.newBuilder()

          def parseFeature(tagName: String, process: (String, String) => Unit) = tag(tagName) = handleComplete {
            val name = attr("name")
            val value = attr("value")
            name match {
              case "star" => builder.setStars(Star.getStar(value).id)
              case _ => process(name, value)
            }
          }

          def addFeature(name: String, value: String): Unit = {
            val (n, v) = if (featureMappings.isDefinedAt((name, value))) {
              featureMappings((name, value))
            } else {
              (name, value)
            }
            builder.addFeaturesBuilder().setName(n).setValue(v)
          }

          def parseSimpleFeature(tagName: String) = parseFeature(tagName, {
            case (name, value) => addFeature(name, value)
          })

          var address = Map.empty[String, Address.Builder].withDefault(lang => Address.newBuilder().setLang(lang))

          complete {
            if (!builder.hasStars) builder.setStars(0)
            builder.addAllAddress(asJavaIterable(address.values.map(_.build()))).setPartner(partner.id)
            Try(builder.build()) match {
              case Success(rawHotel) => success += 1; rawHotel.writeDelimitedTo(os)
              case Failure(e) =>
                log.debug("Can not parse", e)
                failed += 1;
            }
          }

          tag("name") = handleComplete {
            language.foreach { lang =>
              builder.addNameBuilder()
                .setLang(lang)
                .setValue(text)
            }
          }
          tag("url") = handleComplete {
            builder.setHotelUrl(text)
          }
          tag("company-id") = handleComplete {
            builder.setPartnerId(text)
          }
          tag("country") = handleComplete {
            language.foreach { lang =>
              address += lang -> address(lang).setCountry(text)
            }
          }
          tag("locality-name") = handleComplete {
            language.foreach { lang =>
              address += lang -> address(lang).setLocality(text)
            }
          }
          tag("address") = handleComplete {
            language.foreach { lang =>
              address += lang -> address(lang).setFullAddress(text)
            }
          }
          tag("coordinates") = handle {
            val point = builder.getPointBuilder
            tag("lon") = handleComplete(Try(text.toDouble).foreach(point.setLongitude))
            tag("lat") = handleComplete(Try(text.toDouble).foreach(point.setLatitude))
          }
          tag("add-url") = handleComplete {
            builder.setPartnerUrl(text)
          }
          tag("photos") = handle {
            tag("photo") = handle {
              builder.addRawImages(attr("url"))
            }
          }
          tag("phones") = handle {
            tag("phone") = handle {
              tag("formatted") = handle {
                builder.addPhone(text)
              }
            }
          }

          def processAddInfo() = {
            tag("param") = handle {
              val feature = Feature.newBuilder()
              tag("name") = handleComplete {
                feature.setName(text)
              }
              tag("value") = handleComplete {
                feature.setValue(text)
              }
              complete {
                Try(feature.build()).foreach { feature =>
                  builder.addAddInfo(feature)
                  if (regionParam.contains(feature.getName)) {
                    builder.setRegionId(feature.getValue)
                  }
                  if (feature.getName == "rating") {
                    builder.setRating(feature.getValue.toDouble)
                  }
                }
              }
            }
          }
          tag("add-info") = handle(processAddInfo())
          processAddInfo()
          parseFeature("feature-in-units-single", {
            case (name, strValue) =>
              var value = strValue.toDouble
              val unit = attr("unit-value")
              unit match {
                case "kilometer" => value *= 1000
                case "meter" =>
              }
              addFeature(name, value.toInt.toString)
          })
          parseFeature("feature-boolean", {
            case (name, value) =>
              val unifiedValue = value match {
                case "0" => "false"
                case "false" => "false"
                case _ => "true"
              }
              addFeature(name, unifiedValue)
          })
          parseSimpleFeature("feature-enum-single")
          parseSimpleFeature("feature-enum-multiple")
          parseFeature("feature-numeric-single", {
            case (name, strValue) => try {
              addFeature(name, strValue.toDouble.toInt.toString)
            } catch {
              case NonFatal(e) => log.debug(s"Can not parse feature: $name -> $strValue", e)
            }
          })
        }
      }
    }.setLowerCase(true).parse(is)
    log.info(s"$partner hotels parsing finished. Total: $total, success: $success, failed: $failed")
  }
}
