package ru.yandex.tours.indexer.hotels.parsers

import java.io.{File, InputStream, OutputStream}

import ru.yandex.common.xml.{XParser, XTag}
import ru.yandex.tours.indexer.hotels.PartnerHotelParser
import ru.yandex.tours.model.hotels.HotelsHolder.{Feature, HotelType, RawPartnerHotel}
import ru.yandex.tours.model.hotels.{Partners, Star}
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.util.{IO, Logging}
import ru.yandex.tours.util.lang._

import scala.collection.mutable
import scala.concurrent.Future
import scala.util.{Failure, Success, Try}

class OstrovokV3FormatParser extends PartnerHotelParser with Logging {
  /**
   * @param is - input stream to parse
   * @return file, which contains delimited [[RawPartnerHotel]]
   */
  override def parse(is: InputStream): Future[File] = Try {
    IO.usingTmp("ostrovok_format_parser") { os =>
      createParser(is, os)
    }
  }.toFuture

  private def createParser(is: InputStream, os: OutputStream) = {
    var failed = 0
    var success = 0
    var total = 0
    val unknownAmenities = mutable.HashSet.empty[String]
    val unknownKinds = mutable.HashSet.empty[String]
    new XParser {
      override def build(): XTag = new XTag() {
        private val lang = attr("language")
        tag("hotel") = handle {
          total += 1
          val builder = RawPartnerHotel.newBuilder().setPartner(partner.id)
          val address = builder.addAddressBuilder().setLang(lang)
          builder.setPartnerId(attr("code"))
          tag("location") = handleComplete {
            val lon = attr("lon").toDouble
            val lat = attr("lat").toDouble
            if (lon > 180 || lon < -180 || lat > 90 || lat < -90) {
              log.warn(s"Invalid lat or lon: $lat $lon / in hotel ${builder.getPartnerId}")
            } else {
              builder.getPointBuilder.setLongitude(lon)
              builder.getPointBuilder.setLatitude(lat)
            }
          }
          tag("uri") = handleComplete {
            builder.setPartnerUrl(text)
          }
          tag("images") = handle {
            tag("image") = handleComplete {
              builder.addRawImages(text)
            }
          }

          tag("additionalFields") = handle {
            tag("field") = handleComplete {
              attr("name") match {
                case "bookingID" => builder.addAddInfoBuilder().setName("booking").setValue(text)
                case "expediaID" => builder.addAddInfoBuilder().setName("expedia").setValue(text)
              }
            }
          }

          tag("starRating") = handleComplete {
              Try(Star.getStar(text.toInt).id).foreach(builder.setStars)
          }

          tag("type") = handleComplete {
            text match {
              case "Apartment" => builder.setType(HotelType.APARTMENTS)
              case "BNB" => builder.setType(HotelType.HOTEL)
              case "Boutique_and_Design" =>
                builder.setType(HotelType.HOTEL)
                addFeature(builder, "hotel_type", "design_hotel")
              case "Camping" => builder.setType(HotelType.CAMPING)
              case "Castle" => builder.setType(HotelType.HOTEL)
              case "Cottages_and_Houses" => builder.setType(HotelType.VILLA)
              case "Farm" => builder.setType(HotelType.FARMHOUSE)
              case "Guesthouse" => builder.setType(HotelType.GUESTHOUSE)
              case "Hostel" => builder.setType(HotelType.HOSTEL)
              case "Hotel" => builder.setType(HotelType.HOTEL)
              case "Mini-hotel" => builder.setType(HotelType.MINI_HOTEL)
              case "Resort" => builder.setType(HotelType.HOTEL)
              case "Sanatorium" => builder.setType(HotelType.SANATORIUM)
              case "Villas_and_Bungalows" => builder.setType(HotelType.VILLA)
              case x => unknownKinds += x
            }
          }
          tag("country") = handleComplete {
            address.setCountry(text)
          }
          tag("city") = handleComplete {
            address.setLocality(text)
          }
          tag("address") = handleComplete {
            address.setFullAddress(text)
          }
          tag("name") = handleComplete {
            builder.addNameBuilder().setLang(lang).setValue(text)
          }
          tag("regionId") = handleComplete {
            builder.setRegionId(text)
          }

          tag("amenities") = handle {
            if (lang == "ru") {
              val amenitiesHolder = new PoolAmenities()
              tag("property") = handle {
                tag("amenity") = handleComplete {
                  parseAmenities(builder, text, amenitiesHolder, unknownAmenities)
                }
              }
              (amenitiesHolder.indoorPool, amenitiesHolder.outdoorPool) match {
                case (true, true) => addFeature(builder, "pool_type", "outdoor_and_indoor_pool")
                case (false, true) => addFeature(builder, "pool_type", "outdoor_pool")
                case (true, false) => addFeature(builder, "pool_type", "indoor_pool")
                case (false, false) =>
              }
            }
          }

          complete {
            Try(builder.build()) match {
              case Success(rawHotel) =>
                success += 1; rawHotel.writeDelimitedTo(os)
              case Failure(e) =>
                log.warn("Can not parse", e)
                failed += 1;
            }
          }

        }
      }
    }.parse(is)
    if (unknownAmenities.nonEmpty) {
      log.info(s"Unknown amenities during parsing ostrovok: ${unknownAmenities.mkString(",")}")
    }
    if (unknownKinds.nonEmpty) {
      log.info(s"Unknown kinds of ostrovok hotels: ${unknownKinds.mkString(",")}")
    }
    log.info(s"$partner hotels parsing finished. Total: $total, success: $success, failed: $failed")
  }

  private def addFeature(builder: RawPartnerHotel.Builder, name: String, value: String) = {
    builder.addFeatures(Feature.newBuilder().setName(name).setValue(value))
  }

  private def parseAmenities(hotel: RawPartnerHotel.Builder, amenity: String, poolAmenities: PoolAmenities, unknown: mutable.HashSet[String]) = {
    amenity match {
      case "DVD-плеер" =>
      case "VIP-удобства в номере" =>
      case "Wi-Fi" =>
      case "Аквапарк" =>
      case "Аптека" =>
      case "Аптечка первой помощи" =>
      case "Бадминтон" =>
      case "Банк" =>
      case "Банкомат" => addFeature(hotel, "atm", "true")
      case "Баня" => addFeature(hotel, "has_bathhouse", "true")
      case "Бар" => addFeature(hotel, "has_bar", "true")
      case "Бар оборудован для доступа гостей на инвалидных креслах" =>
      case "Бар у бассейна" =>
      case "Бассейн" => addFeature(hotel, "pool", "true")
      case "Бассейн с подогревом" =>
      case "Бесплатная общественная парковка поблизости" => addFeature(hotel, "type_parking", "free_parking")
      case "Бесплатная парковка" => addFeature(hotel, "type_parking", "free_parking")
      case "Бесплатная парковка рядом с отелем" => addFeature(hotel, "type_parking", "free_parking")
      // It's hack. But it is ok for service ourdays :)
      case "Бесплатный Wi-Fi" => addFeature(hotel, "internet_in_hotel", "free_wi_fi_in_the_room")
      case "Бесплатный доступ в интернет" =>
      case "Бесплатный прокат велосипедов" => addFeature(hotel, "rent_a_bike", "true"); addFeature(hotel, "rental", "rent_a_bike")
      case "Бесплатный трансфер" =>
      case "Бесплатный трансфер до аэропорта" =>
      case "Бесплатный трансфер до горнолыжного склона" =>
      case "Бесплатный трансфер до казино" =>
      case "Бесплатный трансфер до пляжа" =>
      case "Бесплатный трансфер до порта" =>
      case "Бесплатный трансфер до торгового центра" =>
      case "Бесплатный трансфер от/до аэропорта" => addFeature(hotel, "transfer", "true")
      case "Бесплатный трансфер от/до жд вокзала" =>
      case "Бесплатный трансфер по окрестностям" =>
      case "Бесплатный чай/кофе" =>
      case "Библиотека" => addFeature(hotel, "library", "true")
      case "Бизнес-центр" => addFeature(hotel, "business_center", "true")
      case "Бильярд" => addFeature(hotel, "billiards", "true")
      case "Боулинг" => addFeature(hotel, "bowling", "true")
      case "Будильник" =>
      case "Ванна" =>
      case "Верховая езда" => addFeature(hotel, "riding", "true")
      case "Виндсерфинг" => addFeature(hotel, "windsurfing", "true")
      case "Винодельческое хозяйство" =>
      case "Возможен полный пансион" =>
      case "Возможен полупансион" =>
      case "Врач" =>
      case "Гладильные принадлежности" => addFeature(hotel, "ironing", "true")
      case "Гладильные услуги" => addFeature(hotel, "ironing", "true")
      case "Дайвинг" => addFeature(hotel, "diving", "true")
      case "Дартс" =>
      case "Дворецкий" =>
      case "Дежурный врач" =>
      case "Детская игровая площадка" => addFeature(hotel, "kids_place", "true")
      case "Детские телеканалы" =>
      case "Детский бассейн" => addFeature(hotel, "kids_pool", "true")
      case "Детский бассейн с подогревом" =>
      case "Детский клуб" =>
      case "Джакузи" =>
      case "Диетическое меню (по запросу)" =>
      case "Дизайн-отель" => addFeature(hotel, "hotel_type", "design_hotel")
      case "Домофон" =>
      case "Доступ в интернет" =>
      case "Доступ на лыжах к отелю" =>
      case "Душ" =>
      case "Душ/Ванна" =>
      case "Завтрак" =>
      case "Завтрак в номер" =>
      case "Завтрак «шведский стол»" =>
      case "Закрытый бассейн с подогревом" =>
      case "Запирающийся шкафчик" =>
      case "Игровая комната" =>
      case "Индивидуальная регистрация заезда и отъезда" =>
      case "Интернет в номере" =>
      case "Йога" =>
      case "Кабельное телевидение" =>
      case "Казино" => addFeature(hotel, "casino", "true")
      case "Камера хранения" => addFeature(hotel, "left_luggage_office", "luggage")
      case "Камин" =>
      case "Караоке" => addFeature(hotel, "karaoke", "true")
      case "Катание на велосипеде" =>
      case "Катание на лыжах" =>
      case "Катание на лыжах за пределами территории отеля" =>
      case "Каток" =>
      case "Кафе" =>
      case "Компьютер" =>
      case "Кондиционер" => addFeature(hotel, "air_conditioning", "true")
      case "Конференц-зал" =>
      case "Круглосуточная стойка регистрации" =>
      case "Круглосуточный спортзал" =>
      case "Крытый бассейн" => poolAmenities.indoorPool = true
      case "Ксерокс" => addFeature(hotel, "photocopying", "true");
      case "Курение запрещено на всей территории" =>
      case "Курение разрешено" =>
      case "Кухня" =>
      case "Лифт" => addFeature(hotel, "elevator", "true")
      case "Лыжная школа" => addFeature(hotel, "ski_school", "true")
      case "Лыжный трансфер" =>
      case "Люкс для новобрачных" => addFeature(hotel, "type_rooms", "newlyweds_room")
      case "Магазины" => addFeature(hotel, "shops", "true")
      case "Массаж" => addFeature(hotel, "massage_services", "true")
      case "Мебель на улице" =>
      case "Места для курения" =>
      case "Микроволновая печь" =>
      case "Мини-бар" =>
      case "Мини-гольф" =>
      case "Мини-кухня" =>
      case "Москитная сетка" =>
      case "Музей" =>
      case "Музыкальный центр" =>
      case "на английском" =>
      case "на испанском" =>
      case "на итальянском" =>
      case "на немецком" =>
      case "на русском" =>
      case "Настольный теннис" => addFeature(hotel, "table_tennis", "true")
      case "Номера для аллергиков" => addFeature(hotel, "type_rooms", "room_allergic")
      case "Номера для курящих" => addFeature(hotel, "type_rooms", "room_for_smokers")
      case "Номера для некурящих" => addFeature(hotel, "type_rooms", "non_smoking_room")
      case "Номера со звукоизоляцией" =>
      case "Ночной клуб" =>
      case "Обмен валюты" => addFeature(hotel, "currency_exchange", "true")
      case "Оборудование для встреч и презентаций" =>
      case "Обслуживание номеров" =>
      case "Общая кухня" =>
      case "Огнетушитель" =>
      case "Оздоровительный клуб" =>
      case "Организация встреч и банкетов" =>
      case "Отель для некурящих" =>
      case "Отель только для взрослых" =>
      case "Открытый бассейн" => poolAmenities.outdoorPool = true
      case "Открытый бассейн с подогревом" => poolAmenities.outdoorPool = true
      case "Отопление" =>
      case "Отсутствие лифтов" =>
      case "Охота" =>
      case "Охрана" =>
      case "Парковка" =>
      case "Парковка (по запросу)" => addFeature(hotel, "type_parking", "advance_booking_parking")
      case "Парковка рядом с отелем" => addFeature(hotel, "car_park", "true")
      case "Паровая баня" =>
      case "Парусный спорт" =>
      case "Патио" =>
      case "Пешеходный туризм" => addFeature(hotel, "walking", "true")
      case "Пинг-понг" =>
      case "Плавание на лодках" => addFeature(hotel, "rental", "rent_a_boat")
      case "Платная парковка" => addFeature(hotel, "type_parking", "paid_parking")
      case "Платная парковка рядом с отелем" => addFeature(hotel, "type_parking", "paid_parking")
      case "Платный доступ в интернет" =>
      case "Платный трансфер" =>
      case "Платный трансфер до горнолыжного склона" =>
      case "Платный трансфер до казино" =>
      case "Платный трансфер до пляжа" =>
      case "Платный трансфер до порта" =>
      case "Платный трансфер до торгового центра" =>
      case "Платный трансфер от/до аэропорта" =>
      case "Платный трансфер от/до жд вокзала" =>
      case "Платный трансфер по окрестностям" =>
      case "Площадка для барбекю" =>
      case "Площадка для пикника" =>
      case "Подходит для проведения праздничных мероприятий" =>
      case "Поздняя регистрация выезда" =>
      case "Поле для гольфа" => addFeature(hotel, "area_golf", "true")
      case "Поле для гольфа (в пределах 3 км)" => addFeature(hotel, "area_golf", "true")
      case "Полотенца для пляжа/бассейна" => addFeature(hotel, "beach_towels", "true")
      case "Помещение для хранения лыж" => addFeature(hotel, "left_luggage_office", "ski")
      case "Постельное белье" =>
      case "Посудомоечная машина" =>
      case "Прачечная" => addFeature(hotel, "laundry", "true")
      case "Пресса" =>
      case "Пресс для брюк" =>
      case "Продажа билетов" =>
      case "Продажа горнолыжных абонементов" =>
      case "Прокат автомобилей" => addFeature(hotel, "rental", "rent_a_car")
      case "Прокат велосипедов" => addFeature(hotel, "rent_a_bike", "true"); addFeature(hotel, "rental", "rent_a_bike")
      case "Прокат видео- и компьютерной техники" =>
      case "Прокат лыжной экипировки" => addFeature(hotel, "rental", "ski_hire")
      case "Прокат снегоходов и квадроциклов" => addFeature(hotel, "rental", "sledge_argamaks_rental")
      case "Работают лифты для доступа к верхним этажам" =>
      case "Радио" =>
      case "Развлекательные мероприятия" =>
      case "Развлечения на воде" =>
      case "Размещение подходит для семей/детей" =>
      case "Размещение с домашними животными" =>
      case "Размещение с домашними животными (бесплатно)" =>
      case "Размещение с домашними животными не допускается" =>
      case "Размещение с животными (за дополнительную плату)" =>
      case "Ранняя регистрация заезда" =>
      case "Ресторан" => addFeature(hotel, "has_restaurant", "true")
      case "Ресторан оборудован для доступа гостей на инвалидных креслах" =>
      case "Ресторан («шведский стол»)" =>
      case "Рыбалка" => addFeature(hotel, "fishing", "true")
      case "Сад" => addFeature(hotel, "garden", "true")
      case "Салон красоты" =>
      case "Сауна" => addFeature(hotel, "sauna", "true")
      case "Сезонный крытый бассейн" => poolAmenities.indoorPool = true
      case "Сезонный открытый бассейн" => poolAmenities.outdoorPool = true
      case "Сейф" =>
      case "Сейф (в номере)" =>
      case "Сейф для ноутбука" =>
      case "Сквош" => addFeature(hotel, "squash", "true")
      case "Сноркелинг" =>
      case "Сноуборд" =>
      case "Снэк-бар" =>
      case "Солярий" => addFeature(hotel, "solarium", "true")
      case "Спа-центр" => addFeature(hotel, "spa_services", "true")
      case "Стиральная машина" =>
      case "Стойка для одежды" =>
      case "Сувенирный магазин" =>
      case "Сушилка" =>
      case "Тапочки" =>
      case "Театр" =>
      case "Телевизор" =>
      case "Телевизор в лобби" =>
      case "Телевизор с плоским экраном" =>
      case "Телефон" =>
      case "Теннисный корт" => addFeature(hotel, "tennis_court", "true")
      case "Терраса" =>
      case "Терраса для загара" =>
      case "Торговый автомат с закусками/напитками" =>
      case "Трансфер до аэропорта (за дополнительную плату)" =>
      case "Трансфер от аэропорта" => addFeature(hotel, "transfer", "true")
      case "Трансфер от аэропорта (за дополнительную плату)" =>
      case "Трансфер от/до аэропорта" =>
      case "Тренажерный зал" =>
      case "Удобства для барбекю" => addFeature(hotel, "bbq", "true")
      case "Удобства для бассейна" =>
      case "Удобства для водных видов спорта" =>
      case "Удобства для гостей с ограниченными физическими возможностями" => addFeature(hotel, "room_disabled", "true")
      case "Удобства для пляжа" =>
      case "Упакованные завтраки/обеды" =>
      case "Упакованные ланчи" =>
      case "Ускоренная регистрация выезда" =>
      case "Ускоренная регистрация заезда" =>
      case "Ускоренная регистрация заезда и выезда" =>
      case "Услуга &#34;звонок - будильник&#34;" =>
      case "Услуги консьержа" =>
      case "Услуги няни и уход за детьми" =>
      case "Утюг" =>
      case "Факс" => addFeature(hotel, "fax", "true")
      case "Факс и ксерокс" => addFeature(hotel, "photocopying", "true"); addFeature(hotel, "fax", "true")
      case "Фен" => addFeature(hotel, "hair_dryer", "true")
      case "Фен (по запросу)" => addFeature(hotel, "hair_dryer", "true")
      case "Фитнес-центр" =>  addFeature(hotel, "fitness_center", "true")
      case "Халат" =>
      case "Халат (по запросу)" =>
      case "Хаммам" => addFeature(hotel, "steam", "turkish_bath")
      case "Химчистка" => addFeature(hotel, "dry_cleaning", "true")
      case "Холодильник" => addFeature(hotel, "refrigerator", "true")
      case "Хранение багажа" =>
      case "Часовня" =>
      case "Частный пляж" => addFeature(hotel, "private_beach", "true")
      case "Чистка обуви" => addFeature(hotel, "shoeshine", "true")
      case "Шкаф/гардероб" =>
      case "Экскурсионное бюро" =>
      case x => unknown += x
    }
  }

  private class PoolAmenities(var outdoorPool: Boolean = false, var indoorPool: Boolean = false)

  override val partner: Partner = Partners.ostrovokv3
}
