package ru.yandex.tours.indexer.hotels.parsers

import java.io.{File, InputStream, InputStreamReader, OutputStream}

import com.google.common.base.Charsets
import ru.yandex.common.xml.{XParser, XTag}
import ru.yandex.tours.geo.Iso2Country
import ru.yandex.tours.geo.base.Region
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.indexer.hotels.PartnerHotelParser
import ru.yandex.tours.indexer.hotels.parsers.booking.BookingHotel2RegionRetriever
import ru.yandex.tours.model.Languages
import ru.yandex.tours.model.hotels.HotelsHolder.{HotelType, RawPartnerHotel}
import ru.yandex.tours.model.hotels.{Partners, Star}
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.partners.BookingHttp
import ru.yandex.tours.util.{IO, Logging}

import scala.collection.mutable
import scala.concurrent.{ExecutionContext, Future}
import scala.util.{Failure, Success, Try}

class SingleFileBookingFormatParser(tree: Tree,
                                    iso2country: Iso2Country,
                                    regionRetriever: BookingHotel2RegionRetriever)(implicit ec: ExecutionContext)
  extends PartnerHotelParser with Logging {

  /**
   * @param is - input stream to parse
   * @return file, which contains delimited [[RawPartnerHotel]]
   */
  override def parse(is: InputStream): Future[File] = {
    regionRetriever.retrieve.map { hotel2region =>
      IO.usingTmp("single_booking") { os =>
        parse(is, os, hotel2region)
      }
    }
  }

  def parse(is: InputStream, os: OutputStream, hotel2region: Map[String, String]): Unit = {
    var failed = 0
    var success = 0
    var total = 0
    val unknownCategories = mutable.HashSet.empty[String]
    val unknownFacilities = mutable.HashSet.empty[String]
    var unknownCountry = 0
    val parser = new XParser {
      override def build(): XTag = new XTag() {
        tag("listing") = handle {
          total += 1
          val builder = RawPartnerHotel.newBuilder().setPartner(partner.id)
          val names = mutable.Buffer.empty[(String, String)]
          val address = mutable.Map.empty[String, String]
          val city = mutable.Map.empty[String, String]
          var hasIndoorPool = false
          var hasOutdoorPool = false
          var country: Option[Region] = None

          tag("id") = handleComplete {
            builder.setPartnerId(text)
          }

          tag("longitude") = handleComplete {
            Try(text.toDouble).foreach(builder.getPointBuilder.setLongitude)
          }
          tag("latitude") = handleComplete {
            Try(text.toDouble).foreach(builder.getPointBuilder.setLatitude)
          }
          tag("phone") = handleComplete {
            getAttr("type") match {
              case Some("main") => builder.addPhone(text)
              case _ =>
            }
          }
          tag("category") = handleComplete {
            attr("type") match {
              case "hotel" => builder.setType(HotelType.HOTEL)
              case "japanese-style business hotel" => builder.setType(HotelType.HOTEL)
              case "aparthotel" => builder.setType(HotelType.APARTHOTEL)
              case "apartment" => builder.setType(HotelType.APARTMENTS)
              case "hostel" => builder.setType(HotelType.HOSTEL)
              case "motel" => builder.setType(HotelType.MOTEL)
              case "camping" => builder.setType(HotelType.CAMPING)
              case "tented camp" => builder.setType(HotelType.CAMPING)
              case "villa" => builder.setType(HotelType.VILLA)
              case "chalet" => builder.setType(HotelType.CHALET)
              case "farm stay" => builder.setType(HotelType.FARMHOUSE)
              case "homestay" => builder.setType(HotelType.HOMESTAY)
              case "guest house" => builder.setType(HotelType.GUESTHOUSE)
              case x => unknownCategories += x
            }
          }
          tag("canonical") = handleComplete {
            getAttr("lang") match {
              case None => builder.setPartnerUrl(text)
              case _ =>
            }
          }
          tag("name") = handleComplete {
            names += getAttr("lang").getOrElse("") -> text
          }
          tag("country") = handleComplete {
            country = iso2country.get(text)
          }
          tag("address") = handle {
            tag("component") = handleComplete {
              val lang = getAttr("lang").getOrElse("")
              attr("name") match {
                case "city" => city += lang -> text
                case "addr1" => address += lang -> text
                case _ =>
              }
            }
          }
          tag("content") = handle {
            tag("images") = handle {
              tag("image") = handleComplete {
                (getAttr("type"), getAttr("url")) match {
                  case (Some(typ), Some(url)) =>
                    if (typ == "main" || typ == "photo") {
                      builder.addRawImages(BookingHttp.increasePhotoQuality(url))
                    }
                  case _ =>
                }
              }
            }
          }
          tag("attributes") = handle {
            tag("attr") = handleComplete {
              getAttr("name") match {
                case Some("num_rooms") => builder.addFeaturesBuilder().setName("room_number").setValue(text)
                case Some("class") => Try(Star.getStar(text.toDouble.toInt).id).foreach(builder.setStars)
                case Some("rating") => Try(text.toDouble).foreach(r => builder.setRating(r * 2d))
                case _ =>
              }
            }
          }
          tag("facilities") = handle {
            tag("facility") = handleComplete {
              getAttr("name").foreach {
                case "child_services" =>
                  builder.addFeaturesBuilder().setName("nursery").setValue("true")
                case "facilities_for_disabled" =>
                  builder.addFeaturesBuilder().setName("room_disabled").setValue("true")
                case "spa_and_wellness_centre" =>
                  builder.addFeaturesBuilder().setName("spa").setValue("true")
                case "elevator" =>
                  builder.addFeaturesBuilder().setName("elevator").setValue("true")
                case "bicycle_rental" =>
                  builder.addFeaturesBuilder().setName("rental").setValue("rent_a_bike")
                case "bar" =>
                  builder.addFeaturesBuilder().setName("has_bar").setValue("true")
                case "airport_shuttle" =>
                  builder.addFeaturesBuilder().setName("transfer").setValue("true")
                case "tennis_court" =>
                  builder.addFeaturesBuilder().setName("tennis_court").setValue("true")
                case "parking" =>
                  builder.addFeaturesBuilder().setName("car_park").setValue("true")
                case "restaurant" =>
                  builder.addFeaturesBuilder().setName("has_restaurant").setValue("true")
                case "pets_allowed" =>
                  builder.addFeaturesBuilder().setName("pets").setValue("true")
                case "family_rooms" =>
                  builder.addFeaturesBuilder().setName("type_rooms").setValue("number_family")
                case "fitness_center" =>
                  builder.addFeaturesBuilder().setName("fitness_center").setValue("true")
                case "non_smoking_rooms" =>
                  builder.addFeaturesBuilder().setName("type_rooms").setValue("non_smoking_room")
                case "indoor_pool" =>
                  hasIndoorPool = true
                case "outdoor_pool" =>
                  hasOutdoorPool = true
                case "wifi" =>
                case "skiing_facilities" =>
                case "terrace" =>
                case "meeting_facilities" =>
                case x => unknownFacilities += x
              }
            }
          }

          complete {
            if (country.isEmpty) {
              unknownCountry += 1
            } else {
              Try {
                val (emptyLang, langed) = names.partition(_._1 == "")
                val used = mutable.Set.empty[(String, String)]
                for {
                  (lang, name) <- langed ++ emptyLang.map { case (_, name) => "en" -> name }
                } if (used.contains((lang, name))) {
                  builder.addSynonymsBuilder().setValue(name).setLang(lang)
                } else {
                  used += lang -> name
                  builder.addNameBuilder().setValue(name).setLang(lang)
                }
                (hasIndoorPool, hasOutdoorPool) match {
                  case (true, true) =>
                    builder.addFeaturesBuilder().setName("pool_type").setValue("outdoor_and_indoor_pool")
                  case (false, true) =>
                    builder.addFeaturesBuilder().setName("pool_type").setValue("outdoor_pool")
                  case (true, false) =>
                    builder.addFeaturesBuilder().setName("pool_type").setValue("indoor_pool")
                  case (false, false) =>
                }
                if (!builder.hasStars) builder.setStars(0)
                hotel2region.get(builder.getPartnerId).foreach(builder.setRegionId)
                setAddress(builder, country.get, city.toMap, address.toMap)
                builder.build()
              } match {
                case Success(rawHotel) => success += 1; rawHotel.writeDelimitedTo(os)
                case Failure(e) =>
                  log.warn("Can not parse", e)
                  failed += 1;
              }
            }
          }
        }

      }
    }
    parser.parse(new InputStreamReader(is, Charsets.UTF_8))
    if (unknownFacilities.nonEmpty) {
      log.info(s"Unknown facilities during parsing booking: ${unknownFacilities.mkString(",")}")
    }
    if (unknownCategories.nonEmpty) {
      log.info(s"Unknown categories of booking hotels: ${unknownCategories.mkString(",")}")
    }
    log.info(s"$partner hotels parsing finished. " +
      s"Total: $total, success: $success, failed: $failed, unknown country: $unknownCountry")
  }

  private def getLangValue(lang: String, map: Map[String, String]): Option[String] = {
    map.get(lang).orElse(map.get("")).orElse(map.headOption.map(_._2))
  }

  private def setAddress(builder: RawPartnerHotel.Builder,
                         country: Region,
                         city: Map[String, String],
                         addresses: Map[String, String]) = {
    var langs = city.keySet ++ addresses.keySet
    if (langs.size == 1 && langs.contains("")) langs = Set("en")
    langs.foreach { lang =>
      val defaultLang = Try(Languages.withName(lang)).getOrElse(Languages.en)
      val address = builder.addAddressBuilder().setLang(defaultLang.toString)
      getLangValue(lang, city).foreach(address.setLocality)
      getLangValue(lang, addresses).foreach(address.setFullAddress)

      address.setCountry(country.name(defaultLang))
    }
  }

  override def partner: Partner = Partners.booking
}
