package ru.yandex.tours.indexer.hotels.parsers.booking

import akka.util.Timeout
import play.api.libs.json.{JsArray, JsObject, Json}
import ru.yandex.tours.partners.{BookingHttp, PartnerConfig}
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.s3.S3Client
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.parsing.Tabbed
import spray.http.StatusCodes

import scala.collection.mutable
import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, Future}

trait BookingHotel2RegionRetriever {
  def retrieve: Future[Map[String, String]]
}

class StaticHotel2RegionRetriever(map: Map[String, String]) extends BookingHotel2RegionRetriever {
  override def retrieve: Future[Map[String, String]] = Future.successful(map)
}

object S3Hotel2RegionRetriever {
  val resourceName = "booking_hotel_2_region.tsv"
}

class S3Hotel2RegionRetriever(s3Client: S3Client)
                             (implicit val ec: ExecutionContext) extends BookingHotel2RegionRetriever {

  override def retrieve: Future[Map[String, String]] = {
    s3Client.get(S3Hotel2RegionRetriever.resourceName).map {
      case (StatusCodes.OK, bytes) =>
        scala.io.Source.fromBytes(bytes).getLines().map {
          case Tabbed(hotelId, regionId) => hotelId -> regionId
        }.toMap
      case (sc, response) => sys.error("Can not download " + S3Hotel2RegionRetriever.resourceName + " from S3" +
        s" Sc: $sc, entity: ${new String(response)}")
    }
  }
}

class HttpHotel2RegionRetriever(httpClient: AsyncHttpClient, config: PartnerConfig)
                               (implicit val ec: ExecutionContext) extends BookingHotel2RegionRetriever with Logging {

  override def retrieve: Future[Map[String, String]] = {
    downloadMapping()
  }

  private def downloadMapping(offset: Int = 0,
                              pageSize: Int = 1000,
                              result: mutable.HashMap[String, String] = mutable.HashMap.empty[String, String]
                               ): Future[Map[String, String]] = {
    log.info(s"$offset booking region mappings downloaded")
    val url = BookingHttp.hotel2region(offset, pageSize)
    httpClient.get(url, config.getSearchHeaders)(new Timeout(10.minute)).flatMap {
      case (StatusCodes.OK, page) =>
        val map = parse(page)
        result ++= map
        if (map.isEmpty) {
          Future.successful(result.toMap)
        } else {
          downloadMapping(offset + pageSize, pageSize, result)
        }
      case (sc, entity) => sys.error(s"Not 200 booking region info response $sc: $entity")
    }
  }

  private def parse(page: String): Map[String, String] = {
    Json.parse(page) match {
      case jsObject: JsObject if (jsObject \ "code").as[String] == "1004" =>
        Map.empty[String, String]
      case JsArray(values) =>
        values.seq.map { value =>
          (for {
            hotelId <- (value \ "hotel_id").asOpt[String]
            cityId <- (value \ "city_id").asOpt[Long]
          } yield hotelId -> cityId.toString)
            .getOrElse(sys.error(s"$value cannot be read"))
        }.toMap
      case _ => sys.error(s"Unexpected json $page")
    }
  }
}
