package ru.yandex.tours.indexer.metro

import java.io.{ByteArrayInputStream, ByteArrayOutputStream, InputStream}
import java.util.zip.GZIPInputStream

import play.api.libs.json.{JsArray, JsObject, Json}
import ru.yandex.tours.model.Agencies.Metro
import ru.yandex.tours.model.util.proto
import ru.yandex.tours.util.IO

//import scala.collection.JavaConversions._
import scala.collection.JavaConverters._
import scala.util.Try

object MetroJsonParser {
  def parseFromJson(source: String): Try[Iterable[Metro]] = Try {
    val json = Json.parse(source)
    val metrosAndLineName = (json \ "cities").as[JsArray].value.flatMap { city =>
      val geoId = (city \ "id").as[Int]
      (city \ "lines").as[JsArray].value.flatMap { line =>
        val color = (line \ "color").as[String]
        val lineName = getName((line \ "name").as[JsObject])
        (line \ "stations").as[JsArray].value.map { station =>
          val id = (station \ "id").as[Int]
          val pointArray = (station \ "ll").as[JsArray]
          val latitude = pointArray(0).as[Double]
          val longitude = pointArray(1).as[Double]

          val transitions = (station \ "transitionsIds").asOpt[JsArray]
            .map(_.as[Seq[Int]])
            .getOrElse(Seq.empty)

          val nameObj = (station \ "name").as[JsObject]
          val names = getName(nameObj).map { case (lang, value) => proto.toLangVal(lang, value) }
          val builder = Metro.newBuilder()
            .setId(id)
            .setGeoId(geoId)
            .setColor(color)
            .addAllName(names.asJava)
            .addAllConnected(transitions.map(Int.box).asJava)

          builder.getPointBuilder
            .setLongitude(longitude)
            .setLatitude(latitude)

          builder → lineName
        }
      }
    }
    // Here we add name of lines to metros with same ids
    val linedMetros = metrosAndLineName.groupBy(_._1.getId).flatMap { case (id, sameMetros) ⇒
      if (sameMetros.size == 1) {
        sameMetros.map(_._1)
      } else {
        sameMetros.map { case (metro, line) ⇒
          val names = metro.getNameList.asScala.map { name ⇒
            val linedName = name.getValue + line.get(name.getLang).map(x => s" ($x)").getOrElse("")
            proto.toLangVal(name.getLang, linedName)
          }
          metro.clearName().addAllName(names.asJava)
        }
      }
    }
    linedMetros.map(_.build)
  }

  private def getName(x: JsObject) = {
    x.value.map { case (lang, name) => lang → clean(name.as[String]) }
  }

  private def clean(name: String) = {
    name.replaceAll("""\(.*?\)""", "").trim
  }

  def jsonIs2is(origIs: InputStream): ByteArrayInputStream = {
    IO.using(origIs) { is =>
      val source = scala.io.Source.fromInputStream(is).getLines().mkString
      val baos = new ByteArrayOutputStream()
      parseFromJson(source).get.foreach(_.writeDelimitedTo(baos))
      new ByteArrayInputStream(baos.toByteArray)
    }
  }
}
