package ru.yandex.tours.indexer.ota

import ru.yandex.extdata.common.meta.DataType
import ru.yandex.extdata.common.service.ExtDataService
import ru.yandex.extdata.loader.engine.DataPersistenceManager
import ru.yandex.tours.indexer.ota.OtaParsers.{Parser, StandardParser, TravelataParser}
import ru.yandex.tours.indexer.task.{AsyncUpdatable, TaskWeight}
import ru.yandex.tours.model.ota.OnlineTourAgency
import ru.yandex.tours.ota.OnlineTourAgencies
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.parsing.{IntValue, Tabbed}
import spray.http.StatusCodes

import scala.concurrent.{ExecutionContext, Future}
import scala.concurrent.duration._
import scala.util.control.NonFatal

abstract class OtaIndexer(agencies: OnlineTourAgencies,
                          updateTime: FiniteDuration,
                          client: AsyncHttpClient,
                          extDataService: ExtDataService,
                          dataPersistenceManager: DataPersistenceManager,
                          name: String)
                         (implicit ec: ExecutionContext)
  extends AsyncUpdatable(updateTime, name) with TaskWeight.Light {

  protected def dataType: DataType

  protected def processId(ota: OnlineTourAgency, id: String): Option[Int]

  protected def getUrl(ota: OnlineTourAgency): String

  protected def createParser(ota: OnlineTourAgency): Parser = ota.billingId match {
    case "Travelata" => new TravelataParser
    case _ => new StandardParser
  }

  override def update: Future[Unit] = {
    download map { mappings =>
      val data = IO.printStream { writer =>
        for {
          operatorResponse <- mappings
          (otaId, id, partnerId) <- operatorResponse
        } {
          writer.println(Tabbed(otaId, id, partnerId))
        }
      }
      dataPersistenceManager.checkAndStore(dataType, data)
    }
  }

  private def download: Future[List[Iterable[(Int, Int, String)]]] = {
    val futures = agencies.agencies.map(downloadPartner)
    Future.sequence(futures)
  }

  private def downloadPartner(ota: OnlineTourAgency): Future[Iterable[(Int, Int, String)]] = {
    val response = client.get(getUrl(ota), ota.headers)(20.seconds)
    val mappings = response map {
      case (StatusCodes.OK, x) =>
        val parser = createParser(ota)
        var failed = 0
        val mapping = parser.getLtIds(x).flatMap { partnerId =>
          val result = processId(ota, partnerId).map(ourId => (ota.id, ourId, partnerId))
          if (result.isEmpty) {
            failed += 1
          }
          result
        } ++ parser.getYandexMapping(x).map {
          case (ourId, partnerId) => (ota.id, ourId, partnerId)
        }
        log.info(s"$name resource fetched from ${ota.name}. $failed unknown object found")
        mapping
      case (sc, x) => throw new Exception(s"Can not download response from ota: ${ota.name}. Sc = $sc, response = $x")
    }
    mappings recover {
      case NonFatal(e) =>
        log.warn(s"Can not update ${ota.name} $name, using old one", e)
        IO.readLines(extDataService.readData(dataType)).collect {
          case Tabbed(IntValue(otaId), IntValue(id), partnerId) if otaId == ota.id =>
            (otaId, id, partnerId)
        }.toVector
    }
  }
}