package ru.yandex.tours.indexer.ratings

import java.io.PrintWriter

import it.unimi.dsi.fastutil.ints.Int2IntOpenHashMap
import ru.yandex.tours.backa.BackaPermalinks
import ru.yandex.extdata.common.service.ExtDataService
import ru.yandex.tours.extdata.DataTypes
import ru.yandex.tours.hotels.{HotelRatings, HotelsIndex}
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.parsing.{IntValue, Tabbed}

import scala.tools.nsc.interpreter.InputStream

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 11.09.15
 */
class RatingsCombiner(hotelsIndex: HotelsIndex, backaPermalinks: BackaPermalinks, extDataService: ExtDataService) {

  private def parseVisits(is: InputStream) = {
    val map = new Int2IntOpenHashMap()
    for (Tabbed(IntValue(hotelId), IntValue(visits)) ← IO.readLines(is)) {
      map.put(hotelId, visits)
    }
    map
  }

  def build: Array[Byte] = {
    val ugcRatings = HotelRatings.parseWithConv(extDataService.readData(DataTypes.hotelUGCRatings), backaPermalinks.reversed)
    val partnerRatings = HotelRatings.parse(extDataService.readData(DataTypes.hotelPartnerRatings))
    val hotelsVisits = parseVisits(extDataService.readData(DataTypes.hotelsVisits))

    def write(writer: PrintWriter, hotelId: Int, ratingHotelId: Int, reviewHotelId: Int, visits: Int): Unit = {
      val rating = ugcRatings.getRating(ratingHotelId)
      val rating2 = if (rating > 0d) rating else partnerRatings.getRating(ratingHotelId)
      val reviewCount = ugcRatings.getReviewsCount(reviewHotelId)

      if (rating2 > 0 || reviewCount > 0) {
        writer.println(Tabbed(
          hotelId,
          rating2,
          reviewCount,
          visits
        ))
      }
    }

    IO.printBytes { writer =>
      for (hotel <- hotelsIndex.hotels) {
        val visits = hotel.partnerIds.map(pi => hotelsVisits.get(pi.travelId)).sum

        for (partnerId <- hotel.partnerIds) {
          write(writer, partnerId.travelId, partnerId.travelId, partnerId.travelId, visits)
        }

        if (ugcRatings.getReviewsCount(hotel.id) == 0 || ugcRatings.getRating(hotel.id) < 1e-3) {
          val ids = hotel.partnerIds.map(_.travelId)
          val ratingId = ids.find(id => ugcRatings.getRating(id) > 0)
            .orElse(ids.find(id => partnerRatings.getRating(id) > 0))
          val reviewId = ids.find(id => ugcRatings.getReviewsCount(id) > 0)
          write(writer, hotel.id, ratingId.getOrElse(hotel.id), reviewId.getOrElse(hotel.id), visits)
        }
      }
    }
  }
}
