package ru.yandex.tours.indexer.resorts

import java.io.{ByteArrayInputStream, InputStream}
import java.net.URI

import akka.util.Timeout
import play.api.libs.json._
import ru.yandex.common.util.date.DateUtil.ThreadLocalDateFormat
import ru.yandex.tours.avatars.AvatarClient
import ru.yandex.tours.direction.Directions
import ru.yandex.tours.model.image.ImageProviders
import ru.yandex.tours.resorts.{Review, VoteTypes}
import ru.yandex.tours.resorts.ReviewsInformation.{Attitude, Author, DataProvider, EReview}
import ru.yandex.tours.util.Logging
import ru.yandex.tours.resorts.Implicits._
import ru.yandex.tours.resorts._

import scala.concurrent.duration._
import scala.concurrent.{Await, ExecutionContext, Future}
import scala.io.Source
import scala.util.Try
import scala.util.control.NonFatal

/**
 * Created by Evgeny Zhoga on 03.12.15.
 */
class SkiResortsProcessor(avatarClient: AvatarClient, directions: Directions)
                         (implicit ec: ExecutionContext) extends Logging {

  private val formatter = new ThreadLocalDateFormat("yyyy-MM-dd HH:mm:ss")
  private val imageLoadTimeout = 30.seconds
  private implicit val timeout = Timeout(imageLoadTimeout)

  def process(is: InputStream): InputStream = {
    val content = Source.fromInputStream(is).mkString
    val json = underscoreToCamelCaseJs(Json.parse(content)).validate[List[SkiInput]]

    json match {
      case JsSuccess(listOfSkies, _) =>
        val listOfResorts = for {
          rawSki <- listOfSkies
          if directions.get(rawSki.id).nonEmpty
        } yield {
          val ski = Await.result(loadImages(rawSki), imageLoadTimeout)
          val mainImage = directions.get(ski.id).map(_.mainImage)
            .map(i => ImageResource(i.provider.toString, i.baseUrlWithoutProtocol))
          val skiWithMainImages = ski.copy(mainImage = mainImage)
          Resort(skiWithMainImages, rawSki.reviews.map(getReviewInformation(ski)))
        }

        new ByteArrayInputStream(camelCaseToUnderscoreJs(Json.toJson(listOfResorts)).toString().getBytes)
      case err@JsError(e) =>
        sys.error(s"Failed to parse: $err")

    }
  }

  private def getReviewInformation(ski: Ski)(reviews: List[Review]): ReviewsInformation = {
    val stats = calcReviewStats(reviews)
    ReviewsInformation(
      resortId = ski.id,
      resortStats = ReviewsInformation.ResortStats(
        resortId = ski.id,
        reviewsCount = stats.reviewsCount,
        scoresCount = stats.scoresCount,
        score = stats.avgScore,
        attitudes = stats.attitudes
      ),
      reviews.map { r =>
        val (url, name) = r.link.flatMap(link => Try(new URI(link)).toOption)
          .map(u => u.getHost -> u.getScheme)
          .map { case (host, scheme) => s"$scheme://$host/" -> host }
          .getOrElse("UNKNOWN" -> "UNKNOWN")
        EReview(
          id = s"agg_${r.id}",
          dtreviewed = formatter.parse(r.date).getTime / 1000,
          resortId = ski.id,
          url = r.link,
          List(Author("NAME", r.userName)),
          r.reviewText.map(SkiResortsProcessor.getSnippet(_)),
          r.stayYear,
          r.stayMonth,
          r.votes.map(v => Attitude(v.`type`, v.rating.map(_.toDouble), v.comment)),
          DataProvider(name, url)
        )
      }.sortBy(_.dtreviewed)(Ordering[Long].reverse)
    )
  }

  private def loadImages(ski: SkiInput): Future[Ski] = {
    ski.imagesUrls.map {
      case urls if urls.nonEmpty =>
        Future.sequence(urls.map { imageUrl =>
          avatarClient.put(imageUrl.url, ImageProviders.ski).map(i => (i, imageUrl.owner))
        }).map { imgs =>
          val imageResources = imgs.map {
            case (image, owner) =>
              ImageResource(
                owner,
                image.baseUrlWithoutProtocol
              )
          }
          ski.asSki.copy(images = Some(imageResources))
        }.recoverWith { case NonFatal(t) =>
          log.warn(s"Image loading failed with exception", t)
          Future.successful(ski.asSki)
        }
      case _ => Future.successful(ski.asSki)
    }.getOrElse(Future.successful(ski.asSki))
  }

  private case class ReviewStats(reviewsCount: Int,
                                 scoresCount: Int,
                                 scoreSum: Double,
                                 votes: Map[String, (Int, Int)]) {
    def avgScore: Option[Double] = if (scoresCount > 0) Some(scoreSum / scoresCount) else None
    def attitudes: List[Attitude] = votes.toList.map {
      case (key, (count, sum)) =>
        Attitude(VoteTypes.withName(key), Some(sum.toDouble / count), None)
    }

    def +(r: Review): ReviewStats = {
      val votesWithRating = r.votes.filter(_.rating.isDefined)

      val avgScore: Option[Double] =
        if (votesWithRating.isEmpty) None
        else Some(votesWithRating.map(_.rating.get).sum.toDouble / votesWithRating.size)

      val newReviewsCount = reviewsCount + r.reviewText.size
      val newScoresCount = scoresCount + avgScore.size
      val newScoreSum = avgScore.fold(scoreSum)(_ + scoreSum)
      val newVotes = votesWithRating.foldLeft(votes) { (acc, next) =>
        val (count, sum) = acc.getOrElse(next.`type`.toString, (0, 0))
        acc + (next.`type`.toString -> (count + 1, sum + next.rating.get))
      }

      ReviewStats(newReviewsCount, newScoresCount, newScoreSum, newVotes)
    }
  }

  private def calcReviewStats(reviews: List[Review]): ReviewStats = {
    val emptyStats = ReviewStats(0, 0, 0.0, Map.empty[String, (Int, Int)])

    reviews.foldLeft(emptyStats) {
      (stats, nextReview) => stats + nextReview
    }
  }
}

object SkiResortsProcessor {
  def getSnippet(text: String, length: Int = 200): String = {
    if (text.length <= length) text.trim
    else {
      val t = text.substring(0, length)
      t.substring(0, t.lastIndexOf(' ')).trim + "..."
    }
  }
}
