package ru.yandex.tours.indexer.sitemap

import java.io._
import java.time.Month
import java.util.zip.{ZipEntry, ZipOutputStream}

import com.google.common.base.Charsets
import com.google.common.xml.XmlEscapers
import org.joda.time.DateTime
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.direction.Thematics
import ru.yandex.tours.model.search.SearchType
import ru.yandex.tours.resorts.SkiResorts
import ru.yandex.tours.util.{IO, Logging}

object SitemapGenerator extends Logging {
  val MAX_FILE_ENTRIES = 50000

  private val escaper = XmlEscapers.xmlContentEscaper()

  def generate(hotelsIndex: HotelsIndex, geoMapping: GeoMappingHolder, skiResorts: SkiResorts): File = {
    val hotelUrls = hotelsIndex.hotels.flatMap { hotel =>
      Seq(s"https://travel.yandex.ru/hotel/${hotel.id}") ++ {
        if (hotel.reviewsCount > 0) Seq(s"https://travel.yandex.ru/hotel/${hotel.id}/reviews")
        else Seq.empty
      }
    }

    val toursUrls = for {
      to <- (geoMapping.countryGeoIds | geoMapping.cityGeoIds).iterator
      if geoMapping.isKnownDestination(to, SearchType.TOURS)
      from <- geoMapping.departuresGeoIds
    } yield s"https://travel.yandex.ru/search?to=$to&from=$from"

    val roomsUrls = for {
      to <- (geoMapping.countryGeoIds | geoMapping.cityGeoIds).iterator
      if geoMapping.isKnownDestination(to, SearchType.ROOMS)
    } yield s"https://travel.yandex.ru/search/rooms?to=$to"


    val thematicUrls = for (thematic <- Thematics.values.iterator if thematic != Thematics.Surfing) yield {
      val res = Seq.newBuilder[String]

      val th = thematic.toString.toLowerCase
      res += s"https://travel.yandex.ru/$th"
      res += s"https://travel.yandex.ru/$th?no_visa=true"

      for (month <- Month.values()) {
        val m = month.getValue
        res += s"https://travel.yandex.ru/$th?month=$m"
        res += s"https://travel.yandex.ru/$th?month=$m&no_visa=true"
      }
      res.result()
    }

    val skiResortsUrls = skiResorts.resorts.iterator.map { resort =>
      s"https://travel.yandex.ru/resort/ski/${resort.ski.id}"
    }
    
    writeFile(
      "tours" -> toursUrls,
      "rooms" -> roomsUrls,
      "thematics" -> thematicUrls.flatten,
      "hotels" -> hotelUrls,
      "resorts" -> skiResortsUrls
    )
  }

  private def writeEntry(zip: ZipOutputStream, name: String)(action: PrintWriter => Unit) = {
    zip.putNextEntry(new ZipEntry(name))
    val pw = new PrintWriter(zip)
    action(pw)
    pw.flush()
    zip.closeEntry()
  }

  def writeFile(allUrls: (String, Iterator[String])*): File = {
    val now = DateTime.now().toString
    val temp = IO.newTempFile("tours-sitemap-", ".zip")
    val zip = new ZipOutputStream(new BufferedOutputStream(new FileOutputStream(temp)), Charsets.UTF_8)

    var parts = 0
    var totalLines = 0
    for ((name, iter) <- allUrls) {
      var lines = 0
      iter.grouped(MAX_FILE_ENTRIES).foreach { urls =>
        val name = s"sitemap.$parts.xml"

        writeEntry(zip, name) { pw =>
          pw.println("""<?xml version="1.0" encoding="UTF-8"?>""")
          pw.println(s"""<!-- Generated at $now -->""")
          pw.println("<urlset xmlns=\"http://www.sitemaps.org/schemas/sitemap/0.9\">")

          for (url <- urls) {
            val escaped = escaper.escape(url)
            pw.println(s"  <url><loc>$escaped</loc></url>")
            lines += 1
          }

          pw.println("</urlset>")
        }

        parts += 1
      }
      totalLines += lines
      log.info(s"Written url group $name with $lines urls")
    }

    log.info(s"Written ${parts + 1} parts with $totalLines urls")

    writeEntry(zip, "sitemap.xml") { pw =>
      pw.println("""<?xml version="1.0" encoding="UTF-8"?>""")
      pw.println(s"""<!-- Generated at $now -->""")
      pw.println("<sitemapindex xmlns=\"http://www.sitemaps.org/schemas/sitemap/0.9\">")

      for (index <- 0 until parts) {
        val url = s"https://travel.yandex.ru/sitemap.$index.xml"
        pw.println(" <sitemap>")
        pw.println(s"  <loc>${escaper.escape(url)}</loc>")
        pw.println(s"  <lastmod>${escaper.escape(now)}</lastmod>")
        pw.println(" </sitemap>")
      }

      pw.println("</sitemapindex>")
    }
    zip.close()
    temp
  }

}
