package ru.yandex.tours.indexer.suggest

import java.io._

import ru.yandex.tours.direction.Priorities
import ru.yandex.tours.geo.base.region.{Tree, Types}
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.model.Languages
import ru.yandex.tours.model.hotels.Hotel
import ru.yandex.tours.search.settings.SearchSettingsHolder
import ru.yandex.tours.util.{IO, Logging}
import ru.yandex.tours.util.collections.Bag
import ru.yandex.tours.util.parsing.Tabbed
import ru.yandex.tours.util.Collections._

import scala.util.control.NonFatal


object SuggestGenerator extends Logging {
  private def mapIs[T](is: InputStream)(processor: String => T) = {
    IO.readLines(is).drop(1).map(processor).toSeq
  }

  def generate(hotels: => TraversableOnce[Hotel],
               geoMapping: GeoMappingHolder,
               tree: Tree,
               countryPriorities: Priorities,
               resortPriorities: Priorities,
               hotelPriority: Priorities,
               searchSettings: SearchSettingsHolder,
               additionalSynonymsIs: InputStream): File = {
    val lang = Languages.ru

    val additionalSynonyms = mapIs(additionalSynonymsIs) { case line =>
      val parts = line.split("\t")
      parts(0).toInt -> parts(2)
    }.toMultiMap.withDefault(_ => Set.empty)

    val tmpFile = IO.newTempFile("suggest", "tmp")
    val pw = new PrintWriter(new FileOutputStream(tmpFile))

    val allCountries = (countryPriorities.all.toSeq.map(_.id) ++ geoMapping.countryGeoIds).distinct
      .map(geoId => geoId -> countryPriorities.getSuggestRank(geoId))
      .sortBy(-_._2)
    for {
      (regionId, suggestRank) <- allCountries
      country <- tree.region(regionId)
      if searchSettings.getRegionSearchSettings(country).isSearchable ||
        regionId == 1056 || regionId == 983 // tours to EGYPT & TURKEY are banned. isSearchable == false
    } {
      val names = (country.allNames ++ additionalSynonyms(regionId)).filter(_.nonEmpty)
      pw.println(Tabbed(
        "country",
        country.name(lang),
        regionId,
        suggestRank + 20000000,
        names.mkString(",")
      ))
    }

    val resortBag = new Bag[String]()
    val allKnownRegions = geoMapping.cityGeoIds ++ searchSettings.allForcedRegionIds ++ resortPriorities.all.map(_.id)
    val resortIds = (allKnownRegions -- geoMapping.countryGeoIds).toSeq
      .map(geoId => geoId -> resortPriorities.getSuggestRank(geoId))
      .sortBy(-_._2)
    for {
      (cityId, priority) <- resortIds
      city <- tree.region(cityId)
      country <- tree.country(city)
      if searchSettings.getRegionSearchSettings(city).isSearchable
    } {
      var ruName = city.name(lang)
      if (city.`type` == Types.CityDistrict) {
        tree.parent(city, Types.City).foreach { parent =>
          ruName = ruName + ", " + parent.name(lang)
        }
      }
      val spaces = " " * resortBag.getAndIncrement(ruName)
      val synonyms = city.allNames ++ additionalSynonyms(cityId) ++ Seq(
        city.name(lang) + " " + country.name(lang),
        city.name(Languages.en) + " " + country.name(Languages.en),
        country.name(lang) + " " + city.name(lang),
        country.name(Languages.en) + " " + city.name(Languages.en)
      ) ++ city.synonyms
      pw.println(Tabbed(
        "resort",
        ruName + spaces,
        cityId,
        priority + 10000000,
        synonyms.filter(_.nonEmpty).mkString(","),
        country.name(lang),
        country.id
      ))
    }

    val hotelBag = new Bag[String]()
    def makeSpaceHack(name: String) = {
      val spaces = " " * hotelBag.getAndIncrement(name)
      name + spaces
    }

    val allRegionNames = tree.regions.flatMap(r => Set(r.name(lang), r.name(Languages.en))).toSet
    for {
      hotel <- hotels
      city <- tree.region(hotel.geoId)
      country <- tree.country(city)
    } {
      val priority = hotelPriority.getSuggestRank(hotel.id)
      val ruName = hotel.name(lang)
      val priorityHack = if (allRegionNames.contains(ruName)) 1 else 0 //HOTELS-2053
      val spaces = " " * hotelBag.getAndIncrement(ruName)
      val synonyms = Set(
        city.name(lang) + " " + country.name(lang),
        city.name(Languages.en) + " " + country.name(Languages.en),
        country.name(lang) + " " + city.name(lang),
        country.name(Languages.en) + " " + city.name(Languages.en),
        city.name(lang),
        city.name(Languages.en),
        country.name(lang),
        country.name(Languages.en)
      ).map(_ + " " + ruName) ++ Iterable(cleanHotelPrefix(ruName)) ++ hotel.synonyms
      pw.println(Tabbed(
        "hotel",
        ruName + spaces,
        hotel.id,
        priority + priorityHack,
        (synonyms.toVector.distinct.map(makeSpaceHack)  ++ country.synonyms).mkString(","),
        city.name(lang),
        city.id,
        country.name(lang),
        country.id
      ))
    }
    pw.close()
    try {
      IO.using(new FileInputStream(tmpFile)) { is =>
        validate(is, geoMapping, hotels.size)
      }
      tmpFile
    } catch {
      case NonFatal(t) ⇒
        IO.deleteFile(tmpFile)
        throw t
    }
  }

  private def validate(is: InputStream, geoMapping: GeoMappingHolder, hotelsSize: Int) = {
    // TODO. Remove 0.5 coefficient after HOTELS-997
    val expectedNumberOfLines = (geoMapping.countryGeoIds.size + geoMapping.cityGeoIds.size + hotelsSize) * 0.5

    var hasEgypt = false
    var hasMaldives = false
    var hasTurkey = false
    var hasSharm = false
    var sharmPriority: Option[Int] = None
    var count = 0
    scala.io.Source.fromInputStream(is).getLines().foreach { line ⇒
      count += 1
      val parts = line.split("\t")
      val `type` = parts(0)
      val id = parts(2).toInt
      val priority = parts(3).toInt
      `type` match {
        case "country" ⇒
          require(parts.size == 5, "Country suggest should have 5 parts")
        case "resort" ⇒
          require(parts.size == 7, "Resort suggest should have 7 parts")
          if (id == SharmElSheikhId) {
            sharmPriority = Some(priority)
          } else {
            sharmPriority.foreach { shP =>
              require(shP > priority, "Sharm-el-Sheikh expected to be most popular resort")
            }
          }
        case "hotel" ⇒
          require(parts.size == 9, "Hotel suggest should have 9 parts")
          sharmPriority.foreach { shP =>
            require(shP > priority, "Sharm-el-Sheikh expected to be more popular than any hotel")
          }
        case x =>
          sys.error(s"Unknown suggest type: $x")
      }
      if (line.startsWith("country\tЕгипет\t1056")) hasEgypt = true
      if (line.startsWith("country\tМальдивы\t10098")) hasMaldives = true
      if (line.startsWith("country\tТурция\t983")) hasTurkey = true
      if (line.startsWith("resort\tШарм-эль-Шейх\t11487")) hasSharm = true
    }

    log.info(s"Generated suggest with $count lines; expected $expectedNumberOfLines")
    require(count >= expectedNumberOfLines, "Too small suggest")
    require(hasEgypt, "No Egypt in suggest")
    require(hasMaldives, "No Maldives in suggest")
    require(hasTurkey, "No Turkey in suggest")
    require(hasSharm, "No Sharm in suggest")
  }

  private val SharmElSheikhId = 11487

  private val ignorePrefix = Set("rooms", "suites", "villas", "виллы", "apartments",
    "studios", "апартаменты", "dormitory", "hotel", "cottages", "bungalows", "коттеджи", "бунгало", "inn",
    "apartment", "home", "villa", "motel", "hostel", "residence", "guesthouse", "plaza", "hôtel", "apartamentos",
    "отель", "cottage", "hostal", "haus", "hotels", "apartamento", "aparthotel", "appartement", "homestay", "résidence",
    "дом", "resorts", "apartament", "гостиница", "хостел", "homes", "flat", "room", "gästehaus", "penzion", "d'hôtes",
    "appartements", "dom", "residences", "apartman", "bungalow", "hotel,", "мини-отель", "hotell", "otel", "mini-hotel")

  private def cleanHotelPrefix(name: String): String = {
    val parts = name.split(" ")
    (if (ignorePrefix.contains(parts(0).toLowerCase)) {
      parts.tail
    } else {
      parts
    }).mkString(" ")
  }
}
