package ru.yandex.tours.indexer.wizard

import java.io.OutputStream

import org.joda.time.LocalDate
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.index.{IndexWriter, WizardIndexing}
import ru.yandex.tours.indexer.SearchIndexUtil._
import ru.yandex.tours.model.BaseModel.Pansion
import ru.yandex.tours.model.hotels.{Hotel, RichHotelSnippet}
import ru.yandex.tours.model.search.SearchProducts.{HotelSnippet, Offer}
import ru.yandex.tours.model.search.{HotelSearchRequest, OfferSearchRequest}
import ru.yandex.tours.util.hotels.HotelSnippetUtil
import ru.yandex.tours.util.lang.Dates._

import scala.collection.JavaConverters._
import scala.collection.mutable

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 27.02.15
 */
class IndexBuilder(tree: Tree, val hotelIndex: HotelsIndex, geoMapping: GeoMappingHolder)
  extends ResourceBuilder {

  private val map = mutable.HashMap.empty[(Int, Int, Int, LocalDate, Int, Int), (Int, Pansion)]
  private val mapDirections = mutable.HashMap.empty[(Int, Int, Int, LocalDate, Int), (Int, Int, Pansion)]
  private val mapHotels = mutable.HashMap.empty[(Int, Int, Int, LocalDate, Int), (Int, Int, Pansion)]

  def +=(request: OfferSearchRequest, offer: Offer, hotel: Hotel): Unit = {
    if (!shouldIndex(request.hotelRequest)) return
    if (!shouldIndex(offer)) return
    val hr = request.hotelRequest
    for (rid <- regionChain(hotel, tree, geoMapping)) {
      add(offer.getSource.getOperatorId, hr.from, rid, offer.getDate.toLocalDate, offer.getNights,
        offer.getHotelId, offer.getPrice, offer.getPansion)
      add(0, hr.from, rid, offer.getDate.toLocalDate, offer.getNights,
        offer.getHotelId, offer.getPrice, offer.getPansion)
      addDirection(offer.getSource.getOperatorId, hr.from, rid, offer.getDate.toLocalDate, offer.getNights,
        offer.getHotelId, offer.getPrice, offer.getPansion)
      addDirection(0, hr.from, rid, offer.getDate.toLocalDate, offer.getNights,
        offer.getHotelId, offer.getPrice, offer.getPansion)
    }
    addHotel(offer.getSource.getOperatorId, hr.from, offer.getDate.toLocalDate, offer.getNights, offer.getHotelId,
      offer.getPrice, offer.getPansion)
    addHotel(0, hr.from, offer.getDate.toLocalDate, offer.getNights, offer.getHotelId, offer.getPrice, offer.getPansion)
  }

  def ++=(request: OfferSearchRequest, offers: Seq[Offer]): Unit = {
    for {
      hotel <- hotelIndex.getHotelById(request.hotelId).toSeq
      offer <- offers
    } this +=(request, offer, hotel)
  }

  def +=(hr: HotelSearchRequest, snippet: HotelSnippet, hotel: Hotel): Unit = {
    if (!shouldIndex(hr)) return
    if (!shouldIndex(snippet)) return

    for {
      pricesPansion <- snippet.getPansionsList.asScala
    } {
      for (rid <- regionChain(hotel, tree, geoMapping)) {
        add(snippet.getSource(0).getOperatorId, hr.from, rid, snippet.getDateMin.toLocalDate, snippet.getNightsMin,
          snippet.getHotelId, pricesPansion.getPrice, pricesPansion.getPansion)

        add(0, hr.from, rid, snippet.getDateMin.toLocalDate, snippet.getNightsMin,
          snippet.getHotelId, pricesPansion.getPrice, pricesPansion.getPansion)

        addDirection(snippet.getSource(0).getOperatorId, hr.from, rid, snippet.getDateMin.toLocalDate,
          snippet.getNightsMin, snippet.getHotelId, pricesPansion.getPrice, pricesPansion.getPansion)

        addDirection(0, hr.from, rid, snippet.getDateMin.toLocalDate, snippet.getNightsMin,
          snippet.getHotelId, pricesPansion.getPrice, pricesPansion.getPansion)
      }

      addHotel(snippet.getSource(0).getOperatorId, hr.from, snippet.getDateMin.toLocalDate, snippet.getNightsMin,
        snippet.getHotelId, pricesPansion.getPrice, pricesPansion.getPansion)

      addHotel(0, hr.from, snippet.getDateMin.toLocalDate, snippet.getNightsMin,
        snippet.getHotelId,  pricesPansion.getPrice, pricesPansion.getPansion)
    }
  }

  def ++=(hr: HotelSearchRequest, snippets: Seq[HotelSnippet]): Unit = {
    for {
      RichHotelSnippet(snippet, hotel) <- HotelSnippetUtil.enrich(snippets, hotelIndex)
    } this +=(hr, snippet, hotel)
  }

/*
  @deprecated("Unused", "22.04.2015")
  def +=(wi: WizardIndexItem): Unit = {
    for (rid <- regionChain(wi.hotelId, hotelIndex, tree, geoMapping)) {
      add(wi.operatorId, wi.from, rid, wi.when.toLocalDate, wi.nights, wi.hotelId, wi.minPrice, Pansion.valueOf(wi.pansion))
      add(0, wi.from, rid, wi.when.toLocalDate, wi.nights, wi.hotelId, wi.minPrice, Pansion.valueOf(wi.pansion))
      addDirection(wi.operatorId, wi.from, WizardIndexing.toDirection(rid), wi.when.toLocalDate, wi.nights, wi.hotelId, wi.minPrice, Pansion.valueOf(wi.pansion))
      addDirection(0, wi.from, WizardIndexing.toDirection(rid), wi.when.toLocalDate, wi.nights, wi.hotelId, wi.minPrice, Pansion.valueOf(wi.pansion))
    }
    addHotel(wi.operatorId, wi.from, wi.when.toLocalDate, wi.nights, wi.hotelId, wi.minPrice, Pansion.valueOf(wi.pansion))
    addHotel(0, wi.from, wi.when.toLocalDate, wi.nights, wi.hotelId, wi.minPrice, Pansion.valueOf(wi.pansion))
  }*/

  private def add(operatorId: Int, from: Int, to: Int, when: LocalDate, nights: Int, hotelId: Int,
                  minPrice: Int, pansion: Pansion): Unit = {
    val key = (operatorId, from, to, when, nights, hotelId)

    def update() = map(key) = minPrice -> pansion

    map.get(key) match {
      case Some((p, pans)) if minPrice <= p => update()
      case Some(pair) => //do nothing
      case None => update()
    }
  }

  private def addDirection(operatorId: Int, from: Int, to: Int, when: LocalDate, nights: Int, hotelId: Int,
                           minPrice: Int, pansion: Pansion): Unit = {
    val key = (operatorId, from, WizardIndexing.toDirection(to), when, nights)

    def update() = mapDirections(key) = (hotelId, minPrice, pansion)

    mapDirections.get(key) match {
      case Some((_, p, _)) if minPrice <= p => update()
      case Some(_) => //do nothing
      case None => update()
    }
  }

  private def addHotel(operatorId: Int, from: Int, when: LocalDate, nights: Int, hotelId: Int,
                       minPrice: Int, pansion: Pansion): Unit = {
    val key = (operatorId, from, WizardIndexing.toHotel(hotelId), when, nights)

    def update() = mapHotels(key) = (hotelId, minPrice, pansion)

    mapHotels.get(key) match {
      case Some((_, p, _)) if minPrice <= p => update()
      case Some(_) => //do nothing
      case None => update()
    }
  }

  def size: Int = map.size

  def write(os: OutputStream): Unit = {
    for (((operatorId, from, to, when, nights), (hotelId, minPrice, pansion)) <- mapDirections) {
      map((operatorId, from, to, when, nights, hotelId)) = (minPrice, pansion)
    }
    for (((operatorId, from, to, when, nights), (hotelId, minPrice, pansion)) <- mapHotels) {
      map((operatorId, from, to, when, nights, hotelId)) = (minPrice, pansion)
    }

    val ordered = map.toArray.sortBy(_._1)

    val writer = new IndexWriter(os)
    try {
      writer.writeHeader(ordered.length)
      for (((operatorId, from, to, when, nights, hotelId), (minPrice, pansion)) <- ordered) {
        writer.writeItem(operatorId, from, to, when, nights, hotelId, minPrice, pansion)
      }
    } finally {
      writer.close()
    }
  }

  def clear(): Unit = {
    map.clear()
    mapDirections.clear()
    mapHotels.clear()
  }
}
