package ru.yandex.tours.indexer.wizard

import java.io.OutputStream

import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.index.WizardIndexing
import ru.yandex.tours.index.WizardIndexing.MicroOffers
import ru.yandex.tours.indexer.SearchIndexUtil
import ru.yandex.tours.model.search.SearchProducts.{HotelSnippet, Offer}
import ru.yandex.tours.model.search.{HotelSearchRequest, OfferSearchRequest}
import ru.yandex.tours.model.wizard.MicroOffer.{HotelMicroOffersProto, MicroOfferProto}
import ru.yandex.tours.util.lang.Dates.RichLocalDate
import ru.yandex.tours.wizard.microoffer.{MicroOfferIndex, MicroOfferWriter}

import scala.collection.JavaConverters._
import scala.collection.mutable

/**
  * Created by asoboll on 03.02.17.
  */
class MicroOfferBuilder(val hotelIndex: HotelsIndex)
  extends ResourceBuilder {

  private val map = mutable.HashMap.empty[Int, HotelMicroOffersProto]

  private def compareAndUpdate(k: Int, v: HotelMicroOffersProto) = {
    map.get(k) match {
      case Some(vOld) if MicroOfferIndex.defaultOrdering.gteq(v, vOld) => None
      case _ => map(k) = v
    }
  }

  def ++=(request: OfferSearchRequest, offers: Seq[Offer]): Unit = {
    this.add(request.hotelRequest, offers, request.hotelId)
  }

  def ++=(hr: HotelSearchRequest, snippets: Seq[HotelSnippet]): Unit = {
    for {
      (hotelId, snippetsById) <- snippets.groupBy(_.getHotelId)
    } this.add(hr, snippetsById.map(_.getSample), hotelId)
  }

  def size: Int = map.size

  def clear(): Unit = map.clear()

  def write(os: OutputStream): Unit = {
    val data = map.values.toSeq.sorted(MicroOfferIndex.idOrdering)
    val writer = MicroOfferWriter(os)
    writer.writeHeader(WizardIndexing.MicroOffers.formatVersion,
      System.currentTimeMillis(), data.size)
    writer.writeItems(data)
    writer.close()
  }

  private def add(request: HotelSearchRequest, offers: Seq[Offer], hotelId: Int): Unit = {
    if (!SearchIndexUtil.shouldIndex(request)) return
    val microOffers = offers
      .filter(shouldIndex)
      .sortBy(_.getPrice)
      .take(MicroOffers.MAX_OFFER_COUNT)
      .map(microOffer)
    if (microOffers.isEmpty) return
    val hotelMicroOffers = HotelMicroOffersProto.newBuilder()
      .setHotelId(hotelId)
      .addAllOffers(microOffers.asJava)
      .setCompactWhen(request.when.toCompactInt)
      .setNights(request.nights)
      .build
    compareAndUpdate(hotelId, hotelMicroOffers)
  }

  private def microOffer(offer: Offer): MicroOfferProto = {
    MicroOfferProto.newBuilder()
      .setOperatorId(offer.getSource.getOperatorId)
      .setRoomName(offer.getRoomType)
      .setPansion(offer.getPansion)
      .setPrice(offer.getPrice)
      .setUrl(offer.getAgentBookingUrl)
      .build
  }

  private def shouldIndex(offer: Offer): Boolean = {
    (!offer.getWithFlight) && offer.getNights > 0
  }
}
