package ru.yandex.tours.indexer.wizard

import java.io.File

import com.google.common.io.Closer
import org.apache.commons.io.FileUtils
import org.apache.commons.io.output.ByteArrayOutputStream
import org.joda.time.DateTime
import ru.yandex.tours.events.SearchEvent
import ru.yandex.tours.events.SearchEvent._
import ru.yandex.tours.util.file._
import ru.yandex.tours.util.{IO, Logging}
import ru.yandex.tours.wizard.resource.WizardResourceLoader.FileShard
import ru.yandex.tours.wizard.resource.{ExpiringWizardResource, RawResource, WizardResourceLoader}

import scala.collection.JavaConverters._
import scala.concurrent.duration.FiniteDuration
import scala.util.control.NonFatal

/**
  * Author: Vladislav Dolbilov (darl@yandex-team.ru)
  * Created: 13.03.15
  */
abstract class ResourceIndexer[R <: ExpiringWizardResource](builder: ResourceBuilder,
                                                            dir: File,
                                                            protected val extension: String,
                                                            maxAge: FiniteDuration) extends Logging {

  require(dir.exists() || dir.mkdirs(), s"[$dir] does not exists")

  def addFreshResource(resource: RawResource): Unit = {
    resource.writeTo(dir / resource.uniqueFileName)
  }

  def buildRawResource(): RawResource = builder.synchronized {
    val os = new ByteArrayOutputStream()
    builder.write(os)
    builder.clear()
    RawResource(os.toByteArray, gzipped = false, extension)
  }

  protected def isActual(event: SearchEvent) = {
    event.eventTime.isAfter(DateTime.now().minus(maxAge.toMillis))
  }

  def push(event: SearchEvent): Unit = {
    if (isActual(event)) {
      builder.synchronized {
        event match {
          case FoundSnippets(time, req, res) => builder ++= (req, res.getHotelSnippetList.asScala)
          case FoundOffers(time, req, res) => builder ++= (req, res.getOfferList.asScala)
          case _: Actualized | _: FoundFlights | _: FoundTransfers => //ignore
          case OstrovokResult(_, _, _, _) => //ignore
        }
      }
    }
  }

  def buildMergedResource(ttl: FiniteDuration, count: Int): Map[Int, File] = {
    val closer = Closer.create()
    val files = (0 until count).map(i => IO.newTempFile(s"tours_wizard_${i}_", extension))
    try {
      val shards = collectAliveShards(ttl).sortBy(_.freshness)(Ordering[Long].reverse)
      shards.foreach(closer.register)

      merger.mergeTo(shards, files)
      files.indices.zip(files).toMap
    } catch {
      case NonFatal(t) =>
        for (file <- files) {
          FileUtils.deleteQuietly(file)
        }
        throw closer.rethrow(t, classOf[Exception])
    } finally {
      closer.close()
    }
  }

  protected def collectAliveShards(ttl: FiniteDuration): Vector[R] = {
    val fileShards = loader.allFiles(dir, extension)
    val (alive, old) = fileShards.partition(_.shard.isAlive(ttl))
    old.foreach { case FileShard(file, shard) =>
      shard.close()
      log.warn(s"Skipped old shard $file created at ${new DateTime(shard.freshness)}")
    }
    alive.map(_.shard)
  }

  protected def loader: WizardResourceLoader[R]

  protected def merger: ResourceMerger[R]
}
