package ru.yandex.tours.indexer.wizard

import java.io.FileInputStream

import org.apache.commons.io.{FileUtils, IOUtils}
import ru.yandex.extdata.loader.engine.DataPersistenceManager
import ru.yandex.tours.extdata.DataTypes.ShardedIndex
import ru.yandex.tours.util.Logging

import scala.concurrent.duration.FiniteDuration

/**
  * Author: Vladislav Dolbilov (darl@yandex-team.ru)
  * Created: 13.03.15
  */
abstract class ResourceScheduler(indexer: ResourceIndexer[_],
                                 distributor: ResourceDistributor,
                                 dataPersistenceManager: DataPersistenceManager,
                                 name: String) extends Logging {

  def doFreshResource(ttl: FiniteDuration): Unit = {
    log.info(s"Building fresh $name")
    val rawResource = indexer.buildRawResource()
    val resource = rawResource.gzip
    log.info(s"Built $name of size ${resource.size}, sha1 = ${resource.hash}")

    log.info(s"Distributing $name")
    distributor.push(resource)
    log.info(s"$name distributed")

    indexer.addFreshResource(resource)
  }

  def doLongResource(ttl: FiniteDuration, count: Int, dt: ShardedIndex): Unit = {
    log.info(s"Building $count long $name-es for last $ttl")
    for ((i, resource) <- indexer.buildMergedResource(ttl, count)) {
      val is = new FileInputStream(resource)
      try {
        log.info(s"Long $name($i) built. Size = " + FileUtils.byteCountToDisplaySize(FileUtils.sizeOf(resource)))
        dataPersistenceManager.checkAndStore(dt(i), is)
      } finally {
        FileUtils.deleteQuietly(resource)
        IOUtils.closeQuietly(is)
      }
    }
  }
}
