package ru.yandex.tours.parsers.common

import java.io.{File, PrintWriter}
import java.nio.file.StandardCopyOption._
import java.nio.file.{Files, Paths}

import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.util.{IO, Logging}
import ru.yandex.tours.util.parsing.{IntValue, Tabbed}

import scala.collection.mutable
import scala.concurrent.Future

class FileStableIds(path: String) extends StableIds with Logging {
  private val partnerId2stable = mutable.HashMap.empty[(Int, String), Int]
  private val knownIds = mutable.HashSet.empty[Int]

  scala.io.Source.fromFile(path).getLines().foreach {
    case Tabbed(IntValue(partnerId), localId, IntValue(stableId)) =>
      partnerId2stable += (partnerId, localId) -> stableId
      knownIds += stableId
  }

  private var currentFreeId = if (knownIds.isEmpty) 1000000 else knownIds.max + 1

  private var idsToPersist = 0

  private def getIdSync(partner: Partner, localId: String): Int = {
    val key = (partner.id, localId)
    partnerId2stable.get(key) match {
      case Some(stableId) => stableId
      case None =>
        idsToPersist += 1
        while (knownIds.contains(currentFreeId)) {
          currentFreeId += 1
        }
        val result = currentFreeId
        currentFreeId += 1
        partnerId2stable += key -> result
        knownIds += result
        result
    }
  }

  override def getId(partner: Partner, localId: String): Future[Int] =
    Future.successful(getIdSync(partner, localId))

  override def persist(): Future[Unit] = Future.successful {
    if (idsToPersist != 0) {
      val newStableIdsFile = IO.newTempFile("stable_ids", ".tsv")
      val pw = new PrintWriter(newStableIdsFile)
      partnerId2stable.foreach { case ((partnerId, localId), ourId) => pw.println(s"$partnerId\t$localId\t$ourId") }
      pw.close()
      log.info(s"New ids added: $idsToPersist")

      Files.move(Paths.get(newStableIdsFile.getAbsolutePath), Paths.get(path), REPLACE_EXISTING)
      idsToPersist = 0
    } else {
      log.info("No new ids to persist")
    }
  }

  override def getIds(partner: Partner): Future[Iterable[(String, Int)]] = Future.successful {
    partnerId2stable.filter(_._1._1 == partner.id).map { case ((_, partnerId), travelId) =>
      partnerId -> travelId
    }
  }

  override def add(partner: Partner, ids: Iterable[String]): Future[Unit] = Future.successful {
    ids.foreach { id =>
      getId(partner, id)
    }
  }
}
