package ru.yandex.tours.stat

import org.joda.time.{DateTime, LocalDate}
import ru.yandex.tours.db.dao.HotelsDao
import ru.yandex.tours.db.model.DbPartnerHotel
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.collections.Bag
import ru.yandex.tours.util.parsing.TSKV

import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, Future}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 15.03.16
 */
class HotelUpdateStatsExport(hotelsDao: HotelsDao, specstatClient: SpecstatClient)(implicit ec: ExecutionContext)
  extends PeriodicStatsExporter("hotel_update_stats") {

  private def isCreatedRecently(hotel: DbPartnerHotel) =
    new DateTime(hotel.created).toLocalDate == new DateTime(hotel.updated).toLocalDate

  def doExport(date: LocalDate): Future[_] = {
    val start = date.toDateTimeAtStartOfDay
    val end = date.plusDays(1).toDateTimeAtStartOfDay
    val created = new Bag[Partner]()
    val updated = new Bag[Partner]()
    val deleted = new Bag[Partner]()

    val query = HotelsDao.UpdatedBetween(start, end)

    hotelsDao.traverse(query) { hotel =>
      val partnerId = hotel.hotel.getRawHotel.getPartner
      Partners.getOpt(partnerId).foreach { partner =>
        if (hotel.isDeleted) deleted += partner
        else if (isCreatedRecently(hotel)) created += partner
        else updated += partner
      }
    }.flatMap { _ =>
      val data = IO.printString { pw =>
        for (partner <- Partners.values) {
          pw.println(TSKV(Seq(
            "fielddate" -> date.toString,
            "partner" -> partner.toString,
            "created" -> created.getCount(partner),
            "updated" -> updated.getCount(partner),
            "deleted" -> deleted.getCount(partner)
          )))
        }
      }
      specstatClient.post("Travel/Hotels/Update/daily", data, 1.hour)
    }
  }

  override def run: Future[_] = {
    doExport(LocalDate.now().minusDays(1))
  }
}
