package ru.yandex.tours.model

import org.joda.time.{DateTime, DateTimeZone, LocalDateTime, LocalTime}
import ru.yandex.tours.model.Agencies.{ProtoAgency, ProtoTimetable}
import ru.yandex.tours.model.Agency.{Timetable, WorkingTimeDescription}
import ru.yandex.tours.model.BaseModel.LangToVal
import ru.yandex.tours.model.geo.MapObject

import scala.collection.JavaConversions._
import scala.collection.JavaConverters._

/* @author berkut@yandex-team.ru */

object Agency {
  val endOfDay = LocalTime.MIDNIGHT.minusMillis(1)
  val twentyForHourInterval = Interval(LocalTime.MIDNIGHT, endOfDay)
  
  private def parseTimetable(proto: ProtoTimetable) = {
    val tz = DateTimeZone.forID(proto.getTimezone)
    val day2intervals = proto.getDay2IntervalsList.map { dayToInterval ⇒
      val intervals = dayToInterval.getIntervalsList.map { interval ⇒
        Interval(LocalTime.fromMillisOfDay(interval.getFrom), LocalTime.fromMillisOfDay(interval.getTo))
      }
      dayToInterval.getDay → intervals
    }.toMap
    Timetable(day2intervals, tz)
  }

  private def parseLocalizedString(x: Iterable[LangToVal]) = {
    LocalizedString(x.toSeq.reverseMap(t ⇒ Languages.withName(t.getLang) → t.getValue).toMap)
  }

  def parseAgency(proto: ProtoAgency): Agency = {
    val timetable = if (proto.hasTimetable) Some(parseTimetable(proto.getTimetable)) else None
    val url = if (proto.hasUrl) Some(proto.getUrl) else None
    Agency(
      proto.getId,
      parseLocalizedString(proto.getNameList),
      proto.getPoint.getLongitude,
      proto.getPoint.getLatitude,
      parseLocalizedString(proto.getAddressList),
      proto.getPhonesList,
      proto.getGeoId,
      url,
      timetable,
      proto.getShows,
      proto.getMetroIdList.asScala
    )
  }

  case class Interval(from: LocalTime, to: LocalTime) {
    def isOpen(x: LocalTime): Boolean = x.isAfter(from) && (x.isBefore(to) || to.equals(LocalTime.MIDNIGHT))

    lazy val is24hour = this == twentyForHourInterval
  }

  private def toLocalDate(now: LocalDateTime, time: LocalTime, dayShift: Int = 0) = {
    now.plusDays(dayShift)
      .withTime(time.getHourOfDay, time.getMinuteOfHour, time.getSecondOfMinute, time.getMillisOfSecond)
  }

  case class Timetable(day2intervals: Map[Int, Seq[Interval]], timeZone: DateTimeZone) {
    def getWorkingTimeDescription(dateTime: DateTime): WorkingTimeDescription = {
      val now = dateTime.toDateTime(timeZone).toLocalDateTime
      val localTime = now.toLocalTime
      val currentDay = now.getDayOfWeek
      val intervals = day2intervals.getOrElse(currentDay, Seq.empty)
      intervals.find(_.isOpen(localTime)) match {
        case None ⇒
          val nextEvent = intervals.find(_.from.isAfter(localTime)) match {
            case Some(nextInterval) ⇒
              toLocalDate(now, nextInterval.from)
            case None ⇒
              var shift = 0
              var interval: Option[Interval] = None
              var nDay = nextDay(currentDay)
              while (interval.isEmpty) {
                shift += 1
                interval = day2intervals.get(nDay).flatMap(_.headOption)
                nDay = nextDay(nDay)
              }
              toLocalDate(now, interval.get.from, shift)
          }
          WorkingTimeDescription(isOpen = false, is24Hour = false, nextEvent)
        case Some(interval) ⇒
          val nextEvent = toLocalDate(now, interval.to)
          val nextInterval = day2intervals.get(nextDay(currentDay))
          val is24hour = interval.is24hour && nextInterval.isDefined && nextInterval.get.exists(_.is24hour)
          WorkingTimeDescription(isOpen = true, is24hour, nextEvent)
      }
    }

    private def nextDay(x: Int) = if (x == 7) 1 else x + 1
  }

  case class WorkingTimeDescription(isOpen: Boolean, is24Hour: Boolean, nextEvent: LocalDateTime)

}

case class Agency(id: Long,
                  names: LocalizedString,
                  longitude: Double,
                  latitude: Double,
                  address: LocalizedString,
                  phones: Iterable[String],
                  geoId: Int,
                  url: Option[String],
                  timetable: Option[Timetable],
                  shows: Int,
                  metroIds: Seq[String]) extends MapObject {

  def currentWorkTimeStatus: Option[WorkingTimeDescription] = {
    timetable.map(_.getWorkingTimeDescription(DateTime.now()))
  }
}
