package ru.yandex.tours.model

import java.util.concurrent.ConcurrentHashMap

import org.apache.commons.codec.digest.DigestUtils
import ru.yandex.tours.model.BaseModel.ProtoImage
import ru.yandex.tours.model.image.ImageFormats.ImageFormat
import ru.yandex.tours.model.image.ImageProviders
import ru.yandex.tours.model.image.ImageProviders.ImageProvider

import scala.collection.JavaConverters._
import scala.util.Try

/* @author berkut@yandex-team.ru */

object Image {
  case class Size(width: Int, height: Int)

  def fromProto(image: ProtoImage): Image = {
    val source = if (image.hasSource) Some(image.getSource) else None
    val size = if (image.hasSize) Some(Size(image.getSize.getWidth, image.getSize.getHeight)) else None
    val pHash = if (image.hasPHash) Some(image.getPHash) else None
    val nnFeatures =
      if (image.getNNetFeaturesCount > 0) Some(image.getNNetFeaturesList.asScala.map(Float.unbox).toArray)
      else if (image.hasNNetFeaturesRaw) {
        Try(image.getNNetFeaturesRaw.split(" ").map(_.trim.toFloat)).toOption
      } else None

    Image(
      host = intern(image.getHost),
      group = image.getGroup,
      provider = ImageProviders(image.getProviderId),
      name = image.getName,
      source = source,
      size = size,
      pHash = pHash,
      nnFeatures = nnFeatures
    )
  }

  def name(orig: String): String = {
    DigestUtils.md5Hex(orig)
  }

  private val hostInternMap = new ConcurrentHashMap[String, String]()
  private def intern(value: String): String = {
    val old = hostInternMap.get(value)
    if (old ne null) return old
    hostInternMap.putIfAbsent(value, value) match {
      case null => value
      case oldValue => oldValue
    }
  }
}

case class Image(host: String,
                 group: Int,
                 provider: ImageProvider,
                 name: String,
                 source: Option[String],
                 size: Option[Image.Size],
                 pHash: Option[String],
                 nnFeatures: Option[Array[Float]]) {

  private def realHost = host.replaceAll("^(https?:)?//", "")

  def toProto: ProtoImage = {
    val builder = ProtoImage.newBuilder()
      .setHost(realHost)
      .setGroup(group)
      .setProviderId(provider.id)
      .setName(name)
    source.foreach(builder.setSource)
    size.foreach(s => builder.getSizeBuilder.setWidth(s.width).setHeight(s.height))
    pHash.foreach(builder.setPHash)
    nnFeatures.foreach(f => builder.addAllNNetFeatures(f.map(Float.box).toSeq.asJava))
    builder.build()
  }

  def isNeedRecover: Boolean = pHash.isEmpty || size.isEmpty || nnFeatures.isEmpty

  def badPhoto: Boolean = size.exists(_.width < 600)

  def baseUrl: String = s"https://$realHost/get-tours/$group/$name"

  def baseUrlWithoutProtocol: String = baseUrl.replaceAll("^https?:", "")

  def inFormat(format: ImageFormat): String = s"$baseUrl/$format"

  def inFormatWithoutProtocol(format: ImageFormat): String = inFormat(format).replaceAll("^https?:", "")
}