package ru.yandex.tours.model

import ru.yandex.tours.model.geo.MapObject

object MapRectangle {
  def byCenterAndSpan(lon: Double, lat: Double, lonSpan: Double, latSpan: Double) = {
    MapRectangle(Math.max(lon - lonSpan / 2, -180), Math.max(lat - latSpan / 2, -90), Math.min(lon + lonSpan / 2, 180), Math.min(lat + latSpan / 2, 90))
  }

  def byBoundaries(minLon: Double, minLat: Double, maxLon: Double, maxLat: Double) = {
    MapRectangle(minLon, minLat, maxLon, maxLat)
  }

  def fromTsv(line: String) = {
    val parts = line.split("\t")
    if (parts.size != 4) throw new Exception(s"4 parts expected for map rectangle, but found: $line")
    val coords = parts.map(_.toDouble)
    MapRectangle.byBoundaries(coords(0), coords(1), coords(2), coords(3))
  }

  def byPoints(points: Iterable[(Double, Double)]) = {
    MapRectangle(
      points.map(_._1).min,
      points.map(_._2).min,
      points.map(_._1).max,
      points.map(_._2).max
    )
  }

  val empty = MapRectangle.byCenterAndSpan(0, 0, 0, 0)
}


/**
 *  Describes rectangle area in Earth surface.
 *  Latitude must be from -90 to 90.
 *  Longitude must be from -180 to 180.
 *  minLon and minLat describes left bottom corner of rectangle
 *  maxLon and maxLat describes right top corner of rectangle
 */
case class MapRectangle(minLon: Double, minLat: Double, maxLon: Double, maxLat: Double) {
  assert(minLat >= -90 && minLat <= 90, "Latitude should be between -90 and 90")
  assert(maxLat >= -90 && maxLat <= 90, "Latitude should be between -90 and 90")
  assert(minLon >= -180 && minLon <= 180, "Longitude should be between -180 and 180")
  assert(maxLon >= -180 && maxLon <= 180, "Longitude should be between -180 and 180")
  assert(minLat <= maxLat, "Invalid boundaries for MapRectangle. Minimal latitude should be less than maximal")

  def toTsv: String = Seq(minLon, minLat, maxLon, maxLat).mkString("\t")

  def isEmpty: Boolean = this.lonSpan < 1e-3 && this.latSpan < 1e-3
  def nonEmpty: Boolean = !this.isEmpty
  def isReversed: Boolean = minLon > maxLon

  private def extend(x: Double, delta: Double, bound: Double) = {
    val result = x + delta
    if (result < -bound) {
      -bound
    } else if (result > bound) {
      bound
    } else {
      result
    }
  }

  def extend(delta: Double): MapRectangle = {
    MapRectangle(
      minLon = extend(minLon, -delta, 180),
      minLat = extend(minLat, -delta, 90),
      maxLon = extend(maxLon, delta, 180),
      maxLat = extend(maxLat, delta, 90)
    )
  }

  def *(a: Double): MapRectangle = {
    MapRectangle.byCenterAndSpan(
      lonCenter, latCenter,
      lonSpan * a, latSpan * a
    )
  }

  def +(rect: MapRectangle): MapRectangle = {
    MapRectangle(
      minLon = minLon min rect.minLon,
      minLat = minLat min rect.minLat,
      maxLon = maxLon max rect.maxLon,
      maxLat = maxLat max rect.maxLat
    )
  }

  def prettyExtend: MapRectangle = {
    (this * 1.8d).extend(0.001d)
  }

  /**
   *
   * @return true if point is in rectangle.
   */
  def contains(lon: Double, lat: Double): Boolean = (if (isReversed) {
    lon >= minLon || lon <= maxLon
  } else {
    lon >= minLon && lon <= maxLon
  }) && lat >= minLat && lat <= maxLat

  def contains(obj: MapObject): Boolean = contains(obj.longitude, obj.latitude)

  def contains(rect: MapRectangle): Boolean = {
    contains(rect.minLon, rect.minLat) && contains(rect.maxLon, rect.maxLat) && isReversed == rect.isReversed
  }

  def intersect(rect: MapRectangle): Boolean = {
    contains(rect) || rect.contains(this) || {
      if (maxLat < rect.minLat || minLat > rect.maxLat || maxLon < rect.minLon || minLon > rect.maxLon) false
      else true
    }
  }

  /**
   * Longitude of center of the rectangle
   */
  lazy val lonCenter = (minLon + maxLon) / 2

  /**
   * Latitude of center of the rectangle
   */
  lazy val latCenter = (minLat + maxLat) / 2

  /**
   * Width of rectangle in degrees
   */
  lazy val lonSpan = Math.abs(maxLon - minLon)

  /**
   * Height of rectangle in degrees
   */
  lazy val latSpan = Math.abs(maxLat - minLat)

  lazy val area = latSpan * lonSpan

  /** (lon, lat) pairs inside rectangle */
  lazy val coordinateIndexes = (if (minLon <= maxLon) {
    minLon.toInt to maxLon.toInt
  } else {
    (minLon.toInt to 180) ++ (-180 to maxLon.toInt)
  }).flatMap(lon => (minLat.toInt to maxLat.toInt).map(lat => (lon, lat)))

  def points: Seq[(Double, Double)] = Seq(
    (minLon, minLat),
    (minLon, maxLat),
    (maxLon, maxLat),
    (maxLon, minLat)
  )

  override def toString: String = s"[[$minLat,$minLon],[$maxLat,$maxLon]]"
}
