package ru.yandex.tours.model

import java.time.Month

import org.apache.commons.lang.StringUtils
import org.joda.time.LocalDate

import scala.Ordering.Implicits.infixOrderingOps

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 01.07.15
 */
sealed trait Seasonality {
  def contains(month: Month): Boolean
  def contains(date: LocalDate): Boolean = contains(Month.of(date.getMonthOfYear))
}

case object UnknownSeasonality extends Seasonality {
  override def contains(month: Month): Boolean = false
}

case object AllYear extends Seasonality {
  override def contains(month: Month): Boolean = true
}

case class MonthInterval(start: Month, end: Month) extends Seasonality {
  override def contains(month: Month): Boolean = {
    val isAfterStart = month >= start
    val isBeforeEnd = month <= end
    if (start <= end) {
      isAfterStart && isBeforeEnd
    } else {
      isAfterStart || isBeforeEnd
    }
  }
}

case class MonthSet(months: Set[Month]) extends Seasonality {
  override def contains(month: Month): Boolean = months.contains(month)
}

case class Combined(seasons: Seasonality*) extends Seasonality {
  override def contains(month: Month): Boolean = seasons.exists(_.contains(month))
}

object Seasonality {
  def apply(seasons: Seasonality*): Seasonality = {
    if (seasons.size == 1) seasons.head
    else if (seasons.isEmpty) UnknownSeasonality
    else Combined(seasons: _*)
  }

  def parse(str: String): Seasonality = {
    str match {
      case "" => UnknownSeasonality
      case "all_year" => AllYear
      case _ if str.contains(",") => Seasonality(str.split(",").map(parse): _*)
      case _ if str.contains("-") =>
        StringUtils.splitPreserveAllTokens(str, '-') match {
          case Array(start, end) => MonthInterval(Month.of(start.toInt), Month.of(end.toInt))
        }
      case _ => MonthSet(Set(Month.of(str.toInt)))
    }
  }

  def serialize(seasonality: Seasonality): String = {
    seasonality match {
      case UnknownSeasonality => ""
      case AllYear => "all_year"
      case MonthInterval(start, end) => start.getValue + "-" + end.getValue
      case MonthSet(months) => months.map(_.getValue).mkString(",")
      case Combined(seasons @_*) => seasons.map(serialize).mkString(",")
    }
  }
}