package ru.yandex.tours.model.hotels

import ru.yandex.tours.model.Languages
import ru.yandex.tours.model.Languages.Lang

/* @author berkut@yandex-team.ru */

object Address {

  def empty(lang: Lang): Address = Address(lang)

  def serialize(addresses: Map[Lang, Address]): String =
    addresses.map(_._2.serialize).mkString(" @@ ")

  def deserialize(addresses: String): Map[Lang, Address] = {
    val parts = addresses.split(" @@ ")
    parts.filter(_.nonEmpty).map(part => {
      val tokens = part.split("\t")
      val lang = Languages.withName(tokenOrEmpty(tokens, 0))
      val country = tokenOrEmpty(tokens, 1)
      val adminName = tokenOrEmpty(tokens, 2)
      val locality = tokenOrEmpty(tokens, 3)
      val subAdminName = tokenOrEmpty(tokens, 4)
      val street = tokenOrEmpty(tokens, 5)
      val house = tokenOrEmpty(tokens, 6)
      val fullAddress = tokenOrEmpty(tokens, 7)
      Address(lang, country, adminName, locality, subAdminName, street, house, fullAddress)
    }).map(t => t.lang -> t).toMap
  }

  private def tokenOrEmpty(x: Array[String], i: Int) = {
    if (i < x.length) {
      x(i)
    } else {
      ""
    }
  }

  def fromProto(address: HotelsHolder.Address): Address = {
    Address(Languages.withName(address.getLang),
      address.getCountry,
      address.getAdminName,
      address.getLocality,
      address.getSubAdminName,
      address.getStreet,
      address.getHouse,
      address.getFullAddress)
  }
}

case class Address(lang: Lang,
                   country: String = "",
                   adminName: String = "",
                   locality: String = "",
                   subAdminName: String = "",
                   street: String = "",
                   house: String = "",
                   fullAddress: String = "") {

  def serialize: String = s"${lang.toString}\t${clean(country)}\t${clean(adminName)}\t${clean(locality)}\t" +
    s"${clean(subAdminName)}\t${clean(street)}\t${clean(house)}\t${clean(fullAddress)}"
  
  private def clean(x: String) = x.replaceAll("\t", " ")

  override def toString: String = {
    Seq(country, adminName, locality, subAdminName, street, house, fullAddress).filterNot(_.isEmpty).mkString(", ")
  }

  def toProto: HotelsHolder.Address = HotelsHolder.Address.newBuilder()
    .setLang(lang.toString)
    .setCountry(country)
    .setAdminName(adminName)
    .setLocality(locality)
    .setSubAdminName(subAdminName)
    .setStreet(street)
    .setHouse(house)
    .setFullAddress(fullAddress)
    .build()
}
