package ru.yandex.tours.model.hotels

import scala.collection.mutable
import scala.reflect._
import scala.util.{Failure, Success, Try}

/* @author berkut@yandex-team.ru */

object Features {
  trait Feature {
    val name: String
  }

  trait FeatureValue {
    val feature: Feature
    val valueAsSet: Set[_]

    def serialize: String
  }

  case class EnumFeature(name: String, values: Set[String]) extends Feature
  case class MultipleEnumFeature(name: String, values: Set[String]) extends Feature
  case class IntFeature(name: String) extends Feature
  case class BooleanFeature(name: String) extends Feature

  case class EnumFeatureValue(feature: EnumFeature, value: Option[String]) extends FeatureValue {
    assert(value.isEmpty || feature.values.contains(value.get))

    def serialize: String = value.map(v => s"${feature.name}:$v").getOrElse("")

    override val valueAsSet = value.toSet
  }

  case class MultipleFeatureValue(feature: MultipleEnumFeature, value: Set[String]) extends FeatureValue {
    for (value <- value) {
      assert(feature.values.contains(value))
    }

    def serialize: String = value.map(v => s"${feature.name}:$v").mkString("\t")

    override val valueAsSet = value.toSet
  }

  case class IntFeatureValue(feature: IntFeature, value: Option[Int]) extends FeatureValue {
    override def serialize: String = value.map(v => s"${feature.name}:$v").getOrElse("")

    override val valueAsSet = value.toSet
  }

  case class BooleanFeatureValue(feature: BooleanFeature, value: Option[Boolean]) extends FeatureValue {
    override val valueAsSet: Set[_] = value.toSet

    override def serialize: String = value.map(v => s"${feature.name}:$v").getOrElse("")
  }

  def serialize(features: Iterable[FeatureValue]): String = {
    features.map(_.serialize).filterNot(_.isEmpty).mkString("\t")
  }

  def parse(name: String, value: String): Option[FeatureValue] = {
    name2feature.get(name).flatMap {
      case t@EnumFeature(_, values) =>
        if (values.contains(value.toLowerCase)) {
          Some(EnumFeatureValue(t, Some(value.toLowerCase)))
        } else {
          None
        }
      case t@MultipleEnumFeature(_, values) =>
        if (values.contains(value.toLowerCase)) {
          Some(MultipleFeatureValue(t, Set(value.toLowerCase)))
        } else {
          None
        }
      case t@IntFeature(_) =>
        Try(value.toInt) match {
          case Success(int) =>
            Some(IntFeatureValue(t, Some(int)))
          case Failure(e) =>
            None
        }
      case t@BooleanFeature(_) =>
        Try(value.toBoolean) match {
          case Success(bool) =>
            Some(BooleanFeatureValue(t, Some(bool)))
          case Failure(e) =>
            None
        }
    }
  }

  def parse(pairs: Seq[HotelsHolder.Feature]): (Iterable[FeatureValue], Iterable[String]) = {
    parse(pairs.map(f ⇒ f.getName -> f.getValue))
  }

  def parse(pairs: Iterable[(String, String)]): (Iterable[FeatureValue], Iterable[String]) = {
    val failures = mutable.Buffer.empty[String]
    val res = mutable.Buffer.empty[FeatureValue]
    for ((name, value) ← pairs) {
      parse(name, value) match {
        case Some(feature) => res += feature
        case None => failures += s"$name:$value"
      }
    }
    val cleanedResult = res.groupBy(_.feature).map(pair => {
      pair._1 match {
        case t@MultipleEnumFeature(_, _) =>
          MultipleFeatureValue(t, pair._2.map(_.asInstanceOf[MultipleFeatureValue].value.head).toSet)
        case _ => pair._2.head
      }
    })
    (cleanedResult, failures)
  }

  def parse(s: String): (Iterable[FeatureValue], Iterable[String]) = {
    if (s.trim.isEmpty) {
      (Iterable.empty, Iterable.empty)
    } else {
      val triples = s.split("\t").map(part => {
        val parts = part.split(":")
        if (parts.size == 2) {
          (parts(0), parts(1))
        } else {
          (part, "")
        }
      })
      parse(triples)
    }
  }
  
  val name2feature = Seq(
    BooleanFeature("kids_pool"),
    BooleanFeature("atm"),
    BooleanFeature("in_room_safe"),
    BooleanFeature("minibar"),
    BooleanFeature("tv_set"),
    BooleanFeature("refrigerator"),
    BooleanFeature("telephone"),
    BooleanFeature("air_conditioning"),
    BooleanFeature("has_restaurant"),
    BooleanFeature("safe_reception"),
    BooleanFeature("internet"),
    BooleanFeature("pool"),
    BooleanFeature("car_park"),
    BooleanFeature("karaoke"),
    BooleanFeature("business_center"),
    BooleanFeature("sauna"),
    BooleanFeature("nursery"),
    BooleanFeature("pets"),
    BooleanFeature("gym"),
    BooleanFeature("massage_services"),
    BooleanFeature("currency_exchange"),
    BooleanFeature("room_disabled"),
    BooleanFeature("barber"),
    BooleanFeature("solarium"),
    BooleanFeature("spa"),
    BooleanFeature("billiards"),
    BooleanFeature("laundry"),
    BooleanFeature("24_front_desk"),
    BooleanFeature("casino"),
    BooleanFeature("library"),
    BooleanFeature("elevator"),
    BooleanFeature("private_beach"),
    BooleanFeature("shops"),
    BooleanFeature("bbq"),
    BooleanFeature("bowling"),
    BooleanFeature("fishing"),
    BooleanFeature("diving"),
    BooleanFeature("snorkeling"),
    BooleanFeature("windsurfing"),
    BooleanFeature("ski_school"),
    BooleanFeature("walking"),
    BooleanFeature("fitness_center"),
    BooleanFeature("tennis_court"),
    BooleanFeature("basketball_court"),
    BooleanFeature("area_golf"),
    BooleanFeature("volleyball_court"),
    BooleanFeature("squash"),
    BooleanFeature("table_tennis"),
    BooleanFeature("animation_staff"),
    BooleanFeature("ironing"),
    BooleanFeature("shoeshine"),
    BooleanFeature("delivery_newspapers"),
    BooleanFeature("nurse"),
    BooleanFeature("tour_desk"),
    BooleanFeature("fax"),
    BooleanFeature("photocopying"),
    BooleanFeature("beauty_salon"),
    BooleanFeature("packed_lunches"),
    BooleanFeature("whirlpool"),
    BooleanFeature("tea_maker"),
    BooleanFeature("coffee_maker"),
    BooleanFeature("hair_dryer"),
    BooleanFeature("microwave"),
    BooleanFeature("extra_beds"),
    BooleanFeature("placement_children"),
    BooleanFeature("garden"),
    BooleanFeature("dry_cleaning"),
    BooleanFeature("riding"),
    BooleanFeature("breakfast_in_room"),
    EnumFeature("hotel_type", Set("mini_hotel", "guest_house", "aparthotel", "suit_hotel", "motel", "hostel", "design_hotel", "spa_hotel", "boutique_hotel", "pension", "eco_hotel", "camping", "grand_hotel", "rotel", "boatel", "floatel", "art_hotel", "business_hotel", "residences_hotel", "hotel_hour")),
    IntFeature("room_number"),
    EnumFeature("pool_type", Set("indoor_pool", "outdoor_pool", "outdoor_and_indoor_pool")),
    MultipleEnumFeature("accepted_credit_cards", Set("visa", "mastercard", "visa_electron", "maestro", "mastercard_electronic", "american_express", "sbercard", "ortcard_international", "zolotaya_korona", "union_card", "jcb_card", "diners_club_international", "stb_card", "cirrus", "eurocard", "china_union_pay")),
    MultipleEnumFeature("type_parking", Set("free_parking", "short_term_parking", "long_term_parking", "outdoor_parking", "covered_parking", "underground_parking", "surface_parking", "guarded_parking", "unguarded_parking", "parking_for_trucks", "parking_for_cars", "vip_parking", "guest_parking", "advance_booking_parking", "paid_parking", "multilevel_parking", "automatic_parking")),
    MultipleEnumFeature("steam", Set("russian_sauna", "turkish_bath", "wood_fired_sauna", "japanese_bath", "finnish_sauna", "infrared_sauna_2", "american_bath", "oxygen_bath", "roman_bath")),
    EnumFeature("clothes_and_shoe_repairs", Set("mending", "shoe_repair", "repair_clothing_and_footwear")),
    MultipleEnumFeature("rental", Set("rent_a_car", "rent_a_bike", "rent_a_boat", "ski_hire", "rental_mountain_skis", "snowboard_rental", "sledge_argamaks_rental", "mini_skis_rental", "rent_snowmobile")),
    MultipleEnumFeature("tv", Set("satellite_tv", "cable_tv")),
    MultipleEnumFeature("type_rooms", Set("standard_room", "room_studio", "villa", "bungalow", "royal_suite", "presidential_suite", "suite", "number_family", "junior", "connecting_rooms", "penthouse", "two_story_room", "newlyweds_room", "room_for_smokers", "non_smoking_room", "room_allergic", "room_for_disabled")),
    MultipleEnumFeature("type_food", Set("without_power", "on_the_menu", "breakfast_buffet", "half_board", "extended_half_board", "full_board", "advanced_full_board", "all_inclusive", "ultra_all_inclusive", "english_breakfast", "american_breakfast", "continental_breakfast", "diet_breakfast")),
    MultipleEnumFeature("left_luggage_office", Set("luggage", "snowboard", "ski")),
    MultipleEnumFeature("internet_in_hotel", Set("free_wi_fi_in_the_room", "free_wi_fi_in_public_areas", "paid_wi_fi_in_the_room", "paid_wi_fi_in_public_areas", "free_cable_internet_in_room", "free_cable_internet_in_public_areas", "paid_cable_internet_in_room", "paid_cable_internet_in_public_areas", "paid_wi_fi_in_some_rooms", "free_wi_fi_in_some_rooms")),
    IntFeature("distance_to_water"),
    IntFeature("distance_slopes"),
    IntFeature("conference_halls"),
    IntFeature("distance_to_center"),
    IntFeature("distance_to_aeroport"),
    IntFeature("number_restaurant"),
    IntFeature("number_bar"),
    IntFeature("number_cafe2"),
    IntFeature("snack_bars"),
    IntFeature("banquet_halls"),
    IntFeature("floors"),
    IntFeature("distance_to_beach"),
    IntFeature("build_year"),
    IntFeature("reconstruct_year"),
    EnumFeature("hotel_line", Set("1", "2", "3")),
    EnumFeature("beach", Set("sand", "stone", "mixed_sand_pebble", "platform", "pebble", "concrete")),
    BooleanFeature("playground"),
    // new features, which is unknown by Backa
    EnumFeature("slope_distance_enum", Set("0", "250", "500", "1000", "1500", "3000")),
    IntFeature("beach_size"),
    IntFeature("territory_size"),
    BooleanFeature("public_beach"),
    BooleanFeature("beach_towels"),
    BooleanFeature("beach_towels_free"),
    BooleanFeature("beach_towels_paid"),
    BooleanFeature("beach_umbrellas"),
    BooleanFeature("beach_umbrellas_free"),
    BooleanFeature("beach_umbrellas_paid"),
    BooleanFeature("beach_sunbeds"),
    BooleanFeature("beach_sunbeds_free"),
    BooleanFeature("beach_sunbeds_paid"),
    BooleanFeature("kids_place"),
    BooleanFeature("kids_animation"),
    BooleanFeature("kitchen"),
    BooleanFeature("spa_services"),
    BooleanFeature("has_conference_halls"),
    BooleanFeature("water_slides"),
    BooleanFeature("has_bar"),
    BooleanFeature("has_bathhouse"),
    BooleanFeature("beach_volleyball_court"),
    BooleanFeature("disco"),
    BooleanFeature("transfer"),
    BooleanFeature("rent_a_bike"),
    BooleanFeature("ski-in")
  ).map(t => t.name -> t).toMap

  // https://wiki.yandex-team.ru/Tourism/201415/Product/HotelFacts#top-20priznakovvporjadkeubyvanijaprioritetasubektivno
  val topFeatures = Seq(
    "private_beach", "hotel_line", "distance_to_beach",
    "beach",
    "internet_in_hotel",
    "distance_to_aeroport", "distance_to_center",
    "pool_type",
    "build_year", "reconstruct_year",
    "room_number", "floors",
    "tennis_court",
    "fitness_center",
    "kids_place", "playground",
    "nursery",
    "table_tennis",
    //parking?
    "tv_set",
    "air_conditioning",
    "refrigerator",
    "tea_maker",
    "hair_dryer",
    "number_bar", "number_restaurant",
    "steam",
    "room_disabled"
  )

  def feature[T : ClassTag](name: String): T = {
    val f = name2feature(name)
    val clazz = classTag[T].runtimeClass
    if (clazz.isInstance(f)) f.asInstanceOf[T]
    else sys.error(s"Cannot cast feature $f to $clazz")
  }

  //to rename to `unify` or `merge`
  def allFeatures(features: Iterable[FeatureValue]): Iterable[FeatureValue] = {
    val map = features.map(t => t.feature.name -> t).toMap
    name2feature.values.map(feature => {
      map.get(feature.name) match {
        case None => feature match {
          case t: EnumFeature => EnumFeatureValue(t, None)
          case t: MultipleEnumFeature => MultipleFeatureValue(t, Set())
          case t: IntFeature => IntFeatureValue(t, None)
          case t: BooleanFeature => BooleanFeatureValue(t, None)
        }
        case Some(value) => value
      }
    })
  }
}
