package ru.yandex.tours.model.hotels

import org.slf4j.LoggerFactory
import ru.yandex.tours.model.Languages.Lang
import ru.yandex.tours.model.geo.MapObject
import ru.yandex.tours.model.hotels.Features.FeatureValue
import ru.yandex.tours.model.hotels.HotelsHolder.{HotelType, TravelHotel}
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.search.SearchType.SearchType
import ru.yandex.tours.model.{Image, Languages, LocalizedString}

import scala.collection.JavaConversions._
import scala.collection.mutable

/* @author berkut@yandex-team.ru */

object Hotel {
  private val log = LoggerFactory.getLogger(super.getClass)

  def parse(x: TravelHotel): (Hotel, Seq[String]) = {
    val warnings = mutable.Buffer.empty[String]
    if (!x.hasPoint) {
      throw new Exception("Hotel without point!")
    }
    val partnerIds = x.getPartnerInfoList.map(PartnerInfo.apply)
    val addresses = x.getAddressList.map(Address.fromProto).map(t => t.lang -> t).toMap
    val names = x.getNameList.map(t => Languages.withName(t.getLang) -> t.getValue).toMap
    val optUrl = if (x.hasHotelUrl) Some(x.getHotelUrl) else None
    val (features, fails) = Features.parse(x.getFeaturesList)
    if (fails.nonEmpty && log.isDebugEnabled) {
      warnings += s"Can not parse features of hotel with id ${x.getId}: ${fails.mkString("\t")}"
    }
    val images = x.getImagesList.map(Image.fromProto)
    val rating = if (x.hasRating) x.getRating else 0
    val reviews = if (x.hasReviewsCount) x.getReviewsCount else 0
    val hotel = Hotel(
      id = x.getId,
      geoId = x.getGeoId,
      longitude = x.getPoint.getLongitude,
      latitude = x.getPoint.getLatitude,
      partnerIds = partnerIds,
      addresses = addresses,
      name = LocalizedString(names),
      synonyms = x.getSynonymsList.map(_.getValue).distinct,
      url = optUrl,
      star = Star.stars(x.getStars),
      features = features,
      rating = rating,
      reviewsCount = reviews,
      images = images,
      phones = x.getPhoneList.toList,
      `type` = x.getType,
      backaPermalink = if (x.hasBackaPermalink) Some(x.getBackaPermalink) else None
    )
    (hotel, warnings)
  }

  def apply(x: TravelHotel): Hotel = {
    val (hotel, warnings) = parse(x)
    warnings.foreach(log.debug)
    hotel
  }

  implicit val ordering = Ordering.ordered[Hotel]

  /**
   *
   * @param rating of hotel
   * @param reviewCount of hotel
   * @return relevance. First, we split into 5 buckets by rating. 0 until 2, 2 until 4, 4 until 6, 6 until 8, 8 to 10.
   *         In the same bucket more relevant hotel has more reviews. In case of equal review count sort by rating.
   */
  def relevance(rating: Double, reviewCount: Int): Double = {
    val bucket = if (rating > 9) {
      4
    } else {
      rating.toInt / 2
    }
    bucket * 100000 + reviewCount * 10 + rating
  }
}

object PartnerInfo {
  def apply(info: HotelsHolder.PartnerInfo): PartnerInfo = {
    val partnerRegion = if (info.hasPartnerRegion) Some(info.getPartnerRegion) else None
    PartnerInfo(Partners(info.getPartner), info.getPartnerId, info.getPartnerUrl, info.getId, partnerRegion)
  }
}

case class PartnerInfo(partner: Partner, id: String, url: String, travelId: Int,
                       partnerRegionId: Option[String] = None) extends Ordered[PartnerInfo] {
  override def compare(that: PartnerInfo): Int = (partner, id).compareTo(that.partner, that.id)
}

case class Hotel(id: Int,
                 geoId: Int,
                 longitude: Double,
                 latitude: Double,
                 partnerIds: Iterable[PartnerInfo],
                 addresses: Map[Lang, Address],
                 name: LocalizedString,
                 synonyms: Seq[String],
                 url: Option[String],
                 star: Star,
                 features: Iterable[Features.FeatureValue],
                 rating: Double,
                 reviewsCount: Int,
                 images: Iterable[Image],
                 phones: Iterable[String] = Iterable.empty,
                 `type`: HotelType = HotelType.HOTEL,
                 backaPermalink: Option[String] = None
                ) extends MapObject with Ordered[Hotel] {

  def searchAvailable(context: SearchType): Boolean = Partners.partnersByType(context).exists(hasPartner)
  def toursSearchAvailable: Boolean = partnerIds.map(_.partner).exists(Partners.isToursPartner)
  def roomsSearchAvailable: Boolean = partnerIds.map(_.partner).exists(Partners.isRoomsPartner)

  override def compare(that: Hotel): Int = {
    this.relevance.compareTo(that.relevance)
  }

  def mainImage: Image = images.headOption.getOrElse(sys.error("No images in hotel: " + id))

  def topFeatures: Seq[FeatureValue] = {
    Features.topFeatures.flatMap { featureName ⇒
      features.find(_.feature.name == featureName)
    }
  }

  def relevance: Double = Hotel.relevance(rating, reviewsCount)
  def hasPartner(partner: Partner): Boolean = partnerIds.exists(_.partner == partner)
  def partnerId(partner: Partner): Iterable[String] = partnerIds.filter(_.partner == partner).map(_.id)
}