package ru.yandex.tours.model.search

import java.util.UUID

import org.apache.commons.codec.digest.DigestUtils
import org.joda.time.LocalDate
import ru.yandex.tours.model.BaseModel.Currency
import ru.yandex.tours.model.Languages
import ru.yandex.tours.model.Languages.Lang
import ru.yandex.tours.model.search.HotelSearchRequest._
import ru.yandex.tours.model.search.SearchRequests.ProtoHotelSearchRequest
import ru.yandex.tours.model.util.Ages
import ru.yandex.tours.model.util.proto._

import scala.collection.JavaConverters._
import scala.collection.immutable.IndexedSeq

/* @author berkut@yandex-team.ru */

object HotelSearchRequest {
  val DATE_SHIFT = 2
  val NIGHTS_SHIFT = 2

  def apply(proto: ProtoHotelSearchRequest): HotelSearchRequest = new HotelSearchRequest(
    proto.getFrom,
    proto.getTo,
    proto.getNights,
    toLocalDate(proto.getWhen),
    proto.getAgesList.asScala.map(Int.unbox),
    proto.getFlexWhen,
    proto.getFlexNights,
    proto.getCurrency,
    Languages.withName(proto.getLang),
    SearchFilter.parse(proto.getFilter)
  )
}

// scalastyle:off covariant.equals
case class HotelSearchRequest(from: Int,
                              to: Int,
                              nights: Int,
                              when: LocalDate,
                              ages: Seq[Int],
                              flexWhen: Boolean = false,
                              flexNights: Boolean = false,
                              currency: Currency = Currency.RUB,
                              lang: Lang = Languages.ru,
                              filter: SearchFilter = EmptySearchFilter,
                              uniqueId: String = UUID.randomUUID().toString,
                              utmSource: Option[String] = Option.empty,
                              utmMedium: Option[String] = Option.empty,
                              utmCampaign: Option[String] = Option.empty,
                              utmContent: Option[String] = Option.empty,
                              utmTerm: Option[String] = Option.empty) extends BaseRequest {

  require(from >= 0, "from region should be positive")
  require(to >= 0, "to region should be positive")

  private val _ages = Ages(ages.toVector)

  def getAges: Ages = _ages
  def kidsAges: Seq[Int] = _ages.kidsAges
  def adults: Int = _ages.adults
  def kids: Int = _ages.kids
  def agesSerializable: Vector[Int] = _ages.serializable
  def kidsWithoutInfants: Int = _ages.kidsWithoutInfants
  def guestsWithoutInfants: Int = _ages.guestsWithoutInfants

  def encoded: String = {
    var result = s"$from:$to:$nights:$when:${agesSerializable.mkString(",")}:$flexWhen:$flexNights"
    if (currency != Currency.RUB) result += (":" + currency.name())
    if (lang != Languages.ru) result += (":" + lang)
    result + filter.encoded
  }
  val sessionId: String = DigestUtils.md5Hex(encoded)
  override def hashCode(): Int = sessionId.hashCode

  override def canEqual(that: Any): Boolean = that.isInstanceOf[HotelSearchRequest]

  override def equals(obj: scala.Any): Boolean = {
    obj match {
      case request: HotelSearchRequest if request.canEqual(this) =>
        request.sessionId == sessionId
      case _ =>
        false
    }
  }

  def nightsRange: IndexedSeq[Int] = {
    val shift = if (flexNights) NIGHTS_SHIFT else 0
    (nights - shift) to (nights + shift) filter (_ > 0)
  }

  def dateRange: IndexedSeq[LocalDate] = {
    val shift = if (flexWhen) DATE_SHIFT else 0
    (-shift to shift).map(i => when.plusDays(i))
  }

  def whenBack: LocalDate = when.plusDays(nights)

  def utmAsMap: Map[String, String] = {
    Map(
      "utm_source" -> utmSource.getOrElse(""),
      "utm_medium" -> utmMedium.getOrElse(""),
      "utm_campaign" -> utmCampaign.getOrElse(""),
      "utm_content" -> utmContent.getOrElse(""),
      "utm_term" -> utmTerm.getOrElse("")
    )
  }

  def asMap: Map[String, Any] = {
    Map(
      "from" -> from,
      "to" -> to,
      "nights" -> nights,
      "when" -> when,
      "ages" -> agesSerializable.mkString(","),
      "when_flex" -> flexWhen,
      "nights_flex" -> flexNights,
      "currency" → currency.name(),
      "lang" → lang.toString,
      SearchFilter.name -> filter.toString,
      "uniqueId" -> uniqueId
    ) ++ utmAsMap
  }

  override def hotelRequest: HotelSearchRequest = this

  def toProto: ProtoHotelSearchRequest = {
    ProtoHotelSearchRequest.newBuilder()
      .setFrom(from)
      .setTo(to)
      .setNights(nights)
      .addAllAges(agesSerializable.map(Int.box).asJava)
      .setWhen(fromLocalDate(when))
      .setFlexNights(flexNights)
      .setFlexWhen(flexWhen)
      .setCurrency(currency)
      .setLang(lang.toString)
      .setFilter(filter.toString)
      .build()
  }

  def extend(subRegionId: Option[Int] = None, hotelIds: Option[List[Int]] = None): ExtendedHotelSearchRequest =
    ExtendedHotelSearchRequest(hotelSearchRequest = this, subRegionId = subRegionId, hotelIds = hotelIds)
}