package ru.yandex.tours.model.search

import com.google.protobuf.ByteString
import org.apache.commons.codec.digest.DigestUtils
import org.joda.time.LocalDate
import ru.yandex.tours.model
import ru.yandex.tours.model.BaseModel.Pansion
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.search.SearchProducts.{Offer, Source}
import ru.yandex.tours.model.util.proto._

import scala.collection.JavaConversions._

object OfferBuilder {

  def build(externalId: String,
            hotelId: Int,
            source: Source,
            when: LocalDate,
            nights: Int,
            pansion: Pansion,
            roomType: String,
            rawPansion: String,
            originalRoomCode: String,
            withTransfer: Boolean,
            withMedicalInsurance: Boolean,
            withFlight: Boolean,
            price: Int,
            links: Iterable[Offer.LinkToPurchase],
            agentBookingUrl: Option[String] = None,
            freeCancellation: Option[Boolean] = None,
            payload: Option[ByteString] = None): Offer = {
    val id = hash(hotelId, source, when, nights, pansion, roomType, withTransfer, withMedicalInsurance, withFlight, freeCancellation)

    val builder = Offer.newBuilder()
      .setId(id)
      .setExternalId(externalId)
      .setHotelId(hotelId)
      .setSource(source)
      .setDate(fromLocalDate(when))
      .setNights(nights)
      .setPansion(pansion)
      .setRoomType(unifyRoomType(roomType))
      .setRawPansion(rawPansion)
      .setWithTransfer(withTransfer)
      .setWithMedicalInsurance(withMedicalInsurance)
      .setWithFlight(withFlight)
      .setPrice(price)
      .setOriginalRoomCode(originalRoomCode)
      .addAllLink(asJavaIterable(links))

    freeCancellation.foreach(builder.setFreeCancellation)
    agentBookingUrl.foreach(builder.setAgentBookingUrl)
    payload.foreach(builder.setPayload)

    builder.build
  }

  def unifyRoomType(name: String): String = {
    if (name.length > 0) {
      (name.head.toUpper + name.tail.toLowerCase).trim
    } else {
      ""
    }
  }

  def hash(hotelId: Int,
           source: Source,
           when: LocalDate,
           nights: Int,
           pansion: Pansion,
           roomType: String,
           withTransfer: Boolean,
           withMedicalInsurance: Boolean,
           withFlight: Boolean,
           freeCancellation: Option[Boolean]): String = {
    val s = Seq(
      hotelId, source.getOperatorId,
      when, nights,
      pansion, roomType,
      withTransfer, withMedicalInsurance, withFlight,
      freeCancellation
    ).mkString(":")

    val operatorPrefix = (if (source.getOperatorId < 10) "0" else "") + source.getOperatorId
    operatorPrefix + "-" + DigestUtils.md5Hex(s)
  }

  def link(provider: model.Source, url: String): Offer.LinkToPurchase = {
    Offer.LinkToPurchase.newBuilder().setBillingId(provider.code).setLink(url).setPartnerName(provider.name).build
  }

  def source(provider: model.Source, partner: Partner): Source = {
    Source.newBuilder()
      .setOperatorId(provider.id)
      .setPartnerId(partner.id)
      .build
  }
}
