package ru.yandex.tours.model.search

import ru.yandex.tours.model.filter.Filter
import ru.yandex.tours.model.filter.hotel.StarFilter
import ru.yandex.tours.model.search.SearchType.SearchType

import scala.collection.immutable.{BitSet, SortedSet}
import scala.util.Try

/**
  * Created by asoboll on 26.04.17.
  */
sealed trait SearchFilter {
  def encoded: String

  override def toString: String = SearchFilter.serialize(this)
}

case object EmptySearchFilter extends SearchFilter {
  override def encoded: String = ""
}

case class HotelSearchFilter(stars: Int) extends SearchFilter {
  require(stars > 0 && stars < 62)

  def encoded: String = ":s" + stars

  val starsSet: SortedSet[Int] = BitSet.fromBitMask(Array(stars))

  def starsFrom: Int = starsSet.head
  def starsTo: Int = starsSet.last
}

object SearchFilter {
  val name: String = "search_filter"

  private def toInt(s: String): Option[Int] = Try(s.toInt).toOption

  def fromFilter(filter: Filter): Option[SearchFilter] = filter match {
    case StarFilter(values) =>
      val starsPacked =
        values.map(_.x).flatMap(toInt)
          .intersect(1 to 5).map(1L << _).foldLeft(0L)(_ | _)
          .toInt
      if (starsPacked == 0 || starsPacked == 62) None
      else Some(HotelSearchFilter(starsPacked))
    case _ => None
  }

  def apply(filters: Iterable[Filter], searchType: SearchType): SearchFilter = {
    searchType match {
      case SearchType.ROOMS =>
        EmptySearchFilter
      case SearchType.TOURS =>
        filters.flatMap(SearchFilter.fromFilter)
          .headOption.getOrElse(EmptySearchFilter)
    }
  }

  private val StarsPattern = "stars(.*)".r

  def parse(s: String): SearchFilter = s match {
    case StarsPattern(stars) => HotelSearchFilter(stars.toInt)
    case _ => EmptySearchFilter
  }

  def serialize(searchFilter: SearchFilter): String = searchFilter match {
    case HotelSearchFilter(stars) => s"stars$stars"
    case EmptySearchFilter => ""
  }
}
