package ru.yandex.tours.model.subscriptions

/**
 * Represents letter that can be sent
 * @param content content of the letter
 * @param sender sender (from) address of the letter
 * @param recipient recipients (to) addresses of the letter
 * @param headers optional additional headers
 */
case class Letter(content: LetterContent,
                  sender: Address,
                  recipient: Recipient,
                  messageId: Option[String] = None,
                  headers: Headers = Headers()) {
  require(content != null, "Null letter content")
  require(sender != null, "Null letter sender")
  require(recipient != null, "Null letter recipient")
}

/**
 * Payload of a letter
 * @param subject subject of the letter
 * @param body body of the letter
 */
case class LetterContent(subject: String,
                         body: Body) {
  require(subject != null && subject.nonEmpty, "Empty letter subject")
  require(body != null, "Null letter body")
}

/** Mail letter body */
sealed trait Body

object Body {

  /** Corresponds to text mail letter body */
  case class Text(value: String) extends Body {
    require(value != null && value.nonEmpty, "Empty text body")
  }

  /** Corresponds to HTML mail letter body */
  case class Html(value: String) extends Body {
    require(value != null && value.nonEmpty, "Empty HTML body")
  }

}

/**
 * Sender or recipient identity.
 * Mirror of InternetMailAddress
 * @param value value of address
 * @param view view (personal) part of the address
 */
case class Address(value: String, view: Option[String] = None) {
  require(value != null && value.nonEmpty, "Empty address value")
}

case class Headers(list: Iterable[(String, String)] = Iterable.empty)

case class Recipient(address: Address, other: Address*) {
  require(address != null, "Null recipient address")
  val all = address :: other.toList
}
