package ru.yandex.tours.model.util

import org.joda.time.LocalDate
import ru.yandex.tours.model.search.HotelSearchRequest

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 05.08.15
 */
sealed trait DateInterval {
  def start: LocalDate
  def end: LocalDate

  def values: Iterator[LocalDate] = Iterator.iterate(start)(_.plusDays(1)).takeWhile(!_.isAfter(end))

  def isInside(date: LocalDate): Boolean = !date.isBefore(start) && !date.isAfter(end)
}

case class FlexDateInterval(when: LocalDate, flexWhen: Boolean) extends DateInterval {
  def dateShift: Int = if (flexWhen) HotelSearchRequest.DATE_SHIFT else 0

  override def start: LocalDate = when.minusDays(dateShift)
  override def end: LocalDate = when.plusDays(dateShift)
}

case class CustomDateInterval(start: LocalDate, end: LocalDate) extends DateInterval {
  require(start.compareTo(end) <= 0, "Start date should be before end date")
}

object DateInterval {
  def apply(start: String, end: String): DateInterval = {
    CustomDateInterval(
      LocalDate.parse(start),
      LocalDate.parse(end)
    )
  }
  def apply(start: LocalDate, end: LocalDate): DateInterval = {
    CustomDateInterval(start, end)
  }
}
