package ru.yandex.tours.model.utm

import java.net.URLEncoder

case class UtmMark(source: String, medium: String, campaign: String, content: String, term: String, query: String,
                   reqId: String) {

  //url safe parameters without leading separator
  lazy val urlSafe = {
    Seq(
      "utm_source" -> source,
      "utm_medium" -> medium,
      "utm_campaign" -> campaign,
      "utm_content" -> content,
      "utm_term" -> term,
      "user_query" -> query,
      "req_id" -> reqId
    ).collect {
      case (name, value) if value.nonEmpty => name + "=" + URLEncoder.encode(value, "UTF-8")
    }.mkString("&")
  }
  def withSource(source: String) = if (source != this.source) copy(source = source) else this
  def withMedium(medium: String) = if (medium != this.medium) copy(medium = medium) else this
  def withCampaign(campaign: String) = if (campaign != this.campaign) copy(campaign = campaign) else this
  def withContent(content: String) = if (content != this.content) copy(content = content) else this
  def withTerm(term: String) = if (term != this.term) copy(term = term) else this
  def withQuery(query: String) = if (query != this.query) copy(query = query) else this
  def withReqId(reqId: String) = if (reqId != this.reqId) copy(reqId = reqId) else this
}

object UtmMark {
  val empty = UtmMark("", "", "", "", "", "", "")

  def apply(source: Option[String] = None,
            medium: Option[String] = None,
            campaign: Option[String] = None,
            content: Option[String] = None,
            term: Option[String] = None,
            query: Option[String] = None,
            reqId: Option[String] = None): UtmMark = {
    new UtmMark(
      source.getOrElse(""),
      medium.getOrElse(""),
      campaign.getOrElse(""),
      content.getOrElse(""),
      term.getOrElse(""),
      query.getOrElse(""),
      reqId.getOrElse("")
    )
  }
}