package ru.yandex.tours.hot

import java.util.Locale

import org.joda.time.LocalDate
import org.joda.time.format.DateTimeFormat
import ru.yandex.tours.direction.HotDirections
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.model.search.{SearchDates, SearchType}
import ru.yandex.tours.model.subscriptions._
import ru.yandex.tours.model.util.{CustomDateInterval, CustomNights, SortType}
import ru.yandex.tours.model.utm.UtmMark
import ru.yandex.tours.prices.{DirectionPrice, PriceSearcher}
import ru.yandex.tours.search.Defaults
import ru.yandex.tours.search.settings.SearchSettingsHolder
import ru.yandex.tours.serialize.UrlBuilder
import ru.yandex.tours.subscriptions.sender.MailSender
import ru.yandex.tours.util.Randoms._
import ru.yandex.tours.util.Speller
import ru.yandex.vertis.scheduler.model.Schedule.{AtFixedHourOfDay, EveryHours}
import ru.yandex.vertis.scheduler.model.{Payload, Task, TaskDescriptor}

import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, Future}

/**
  * Author: Vladislav Dolbilov (darl@yandex-team.ru)
  * Created: 30.05.17
  */
class HotOfferScheduler(priceSearcher: PriceSearcher,
                        mailSender: MailSender,
                        tree: Tree,
                        geoMapping: GeoMappingHolder,
                        searchSettings: SearchSettingsHolder,
                        urlBuilder: UrlBuilder,
                        hotDirections: HotDirections)
                       (implicit ec: ExecutionContext) {

  private val fromAddress = Address("travel-subscriptions@yandex-team.ru", Some("Яндекс.Путешествия"))
  private val dateFormat = DateTimeFormat.mediumDate().withLocale(Locale.forLanguageTag("ru-RU"))

  def sendHotOffers(from: Int): Future[Unit] = {
    val now = LocalDate.now

    val bestPrices = priceSearcher.searchLastDirectionPrices(
      from,
      Defaults.TWO_ADULTS,
      SearchDates(
        CustomDateInterval(now.plusDays(1), now.plusDays(63)),
        CustomNights(5 to 14),
        whenBack = Option.empty
      ),
      filters = Seq.empty,
      freshness = 3.hours
    )
    for {
      prices ← bestPrices
      letter ← renderLetter(from, prices)
    } yield {
      mailSender.send(letter)
    }
  }

  def sendRandomOffer(from: Int): Future[Unit] = {
    val now = LocalDate.now

    val bestPrices = priceSearcher.searchLastDirectionPrices(
      from,
      Defaults.TWO_ADULTS,
      SearchDates(
        CustomDateInterval(now.plusDays(1), now.plusDays(63)),
        CustomNights(5 to 14),
        whenBack = Option.empty
      ),
      filters = Seq.empty,
      freshness = 3.hours
    )
    for {
      prices ← bestPrices
      letter ← renderLetterSingle(from, prices)
    } yield {
      mailSender.send(letter)
    }
  }

  def renderLetter(from: Int, prices: Seq[DirectionPrice]): Future[Letter] = {
    val builder = StringBuilder.newBuilder
    prices
      .filter(p ⇒ hotDirections.isHot(p.to, p.price))
      .filter(p ⇒ tree.region(p.to).isDefined)
      .filter(p ⇒ tree.country(p.to).isDefined)
      .filter(p ⇒ geoMapping.isKnownDestination(p.to))
      .filter(p ⇒ searchSettings.getRegionSearchSettings(p.to).isAllowed(SearchType.TOURS))
      .sortBy(_.price)
      .foreach {
        price ⇒
          val region = tree.region(price.to).get
          val country = tree.country(price.to).get
          val hotDirection = hotDirections.map(price.to)
          val regionName = hotDirection.regionName.getOrElse(region.name.ruName)
          val countryNameOpt = hotDirection.countryName.orElse {
            if (region != country) Some(country.name.ruName) else None
          }
          val url = urlBuilder.forSearch(
            from,
            region.id,
            price.when,
            price.nights,
            context = SearchType.TOURS,
            utm = UtmMark.empty.withSource("hot_offers"),
            sortType = Some(SortType.PRICE_ASC)
          )

          builder ++= "<span>"

          builder ++= regionName
          builder ++= countryNameOpt.fold(" ")(countryName => s", $countryName ")

          builder ++= dateFormat.print(price.when)

          builder ++= " на "
          builder ++= Speller.nights(price.nights)

          builder ++= ": <a href=\""
          builder ++= url
          builder ++= "\">от "
          builder append price.price
          builder ++= "&nbsp;₽</a><br/>"
          builder ++= "</span>"

      }

    val fromRegion = tree.region(from).get

    Future.successful(
      Letter(
        LetterContent(
          subject = s"Горящие предложения из ${fromRegion.genitive} от Яндекс.Путешествий",
          Body.Html(builder.toString())
        ),
        fromAddress,
        Recipient(
          Address("vgorovoy@yandex-team.ru"),
          Address("hot-tours@yandex-team.ru")
        )
      )
    )
  }

  def renderLetterSingle(from: Int, prices: Seq[DirectionPrice]): Future[Letter] = {
    val builder = StringBuilder.newBuilder
    prices
      .filter(p ⇒ hotDirections.isHot(p.to, p.price))
      .filter(p ⇒ tree.region(p.to).isDefined)
      .filter(p ⇒ tree.country(p.to).isDefined)
      .filter(p ⇒ geoMapping.isKnownDestination(p.to))
      .filter(p ⇒ searchSettings.getRegionSearchSettings(p.to).isAllowed(SearchType.TOURS))
      .sample(1)
      .foreach {
        price ⇒
          val region = tree.region(price.to).get
          val country = tree.country(price.to).get
          val url = urlBuilder.forSearch(
            from,
            region.id,
            price.when,
            price.nights,
            context = SearchType.TOURS,
            utm = UtmMark.empty.withSource("hot_offer"),
            sortType = Some(SortType.PRICE_ASC)
          )

          builder ++= "<h2>Выгодный тур в "

          builder ++= region.accusative
          if (region != country) {
            builder ++= " ("
            builder ++= country.name.ruName
            builder ++= ")"
          }
          builder ++= "</h2>"

          builder ++= "Вылет "
          builder ++= dateFormat.print(price.when)
          builder ++= "<br/>"

          builder ++= "<a href=\""
          builder ++= url
          builder ++= "\">от "
          builder append price.price
          builder ++= "&nbsp;₽</a> за двоих на "
          builder ++= Speller.nights(price.nights)


      }

    val fromRegion = tree.region(from).get

    Future.successful(
      Letter(
        LetterContent(
          subject = s"Тур дня из ${fromRegion.genitive} от Яндекс.Путешествий",
          Body.Html(builder.toString())
        ),
        fromAddress,
        Recipient(
          Address("darl@yandex-team.ru"),
          Address("vgorovoy@yandex-team.ru"),
          Address("hot-tours@yandex-team.ru")
        )
      )
    )
  }

  def task: Task = {
    Task(
      TaskDescriptor("hot_offers", EveryHours(3)),
      Payload.Async(() => {
        for {
          _ ← sendHotOffers(213)
          _ ← sendHotOffers(2)
        } yield ()
      })
    )
  }

  def randomHotOffer: Task = {
    Task(
      TaskDescriptor("random_hot_offer", AtFixedHourOfDay(12)),
      Payload.Async(() => {
        for {
          _ ← sendRandomOffer(213)
          _ ← sendRandomOffer(2)
        } yield ()
      })
    )
  }

}
