package ru.yandex.tours.notifier

import akka.actor.Props
import ru.yandex.common.tokenization.TokensDistributor
import ru.yandex.tours.app._
import ru.yandex.tours.clickhouse.ClickHouseSupport
import ru.yandex.tours.direction.HotDirections
import ru.yandex.tours.extdata.ExtDataSupport
import ru.yandex.tours.hot.HotOfferScheduler
import ru.yandex.tours.prices.ClickHousePriceSearcher
import ru.yandex.tours.storage.subscriptions.{NotificationStorageSupport, SubscriptionsStorageSupport}
import ru.yandex.tours.subscriptions.render.HttpRenderer
import ru.yandex.tours.subscriptions.sender.{InstrumentedMailSender, JavaMailSender, MailSender, MonitoredMailSender}
import ru.yandex.tours.subscriptions.{DefaultNotificationBuilder, InstrumentedNotificationBuilder, MonitoredNotificationBuilder, RootSubscriptionActor}
import ru.yandex.vertis.scheduler.model.Task

import scala.concurrent.duration._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 10.08.15
 */
trait NotifierComponents extends SchedulerTasks {
  self: Application
    with AkkaSupport
    with SpraySupport
    with CommonDataHolders
    with ExtDataSupport
    with HttpSupport
    with ZookeeperSupport
    with ClickHouseSupport
    with SubscriptionsStorageSupport
    with NotificationStorageSupport =>

  import akkaSystem.dispatcher

  val priceSearcher = new ClickHousePriceSearcher(clickHouseClient)(dispatcher)

  val notificationBuilder = {
    val builder = new DefaultNotificationBuilder(priceSearcher, hotelsIndex, tourOperators, regionTree)
    val instrumented = new InstrumentedNotificationBuilder(builder)
    new MonitoredNotificationBuilder(instrumented)
  }

  val renderer = new HttpRenderer(httpClient,
    config.getString("tours.subscriptions.renderer.host"),
    config.getInt("tours.subscriptions.renderer.port"),
    hotelsIndex, hotelsVideo,
    geoMappingHolder,
    searchSettings,
    regionTree,
    directions,
    urlBuilder
  )

  val mailSender: MailSender = {
    val sender = new JavaMailSender(config.getString("tours.notifier.smtp-host"))
    val instrumented = new InstrumentedMailSender(sender)
    new MonitoredMailSender(instrumented)
  }

  lazy val rootSubscriptionsActor = akkaSystem.actorOf(RootSubscriptionActor.props(
    zkClientWithComponentNamespace,
    TokensDistributor.Config(distributePeriod = 300.millis, subscribePeriod = 1.minute, printOwnershipsPeriod = None),
    notificationStorage,
    notificationBuilder,
    renderer,
    mailSender
  ), "subscriptions")

  lazy val notificationScheduler = new NotifierScheduler(
    subscriptionStorage,
    rootSubscriptionsActor
  )(akkaSystem.dispatcher)

  lazy val hotOfferScheduler = new HotOfferScheduler(
    priceSearcher, mailSender, regionTree, geoMappingHolder, searchSettings, urlBuilder, hotDirections
  )

  override def schedulingTasks: Seq[Task] = super.schedulingTasks ++ Seq(
    notificationScheduler.task,
    notificationScheduler.disableTask,
    hotOfferScheduler.task,
    hotOfferScheduler.randomHotOffer
  )

  onStart {
    startServer(Props.empty, config.getInt("component.port"))
  }

}
