package ru.yandex.tours.notifier

import akka.actor.ActorRef
import org.joda.time.LocalDate
import ru.yandex.tours.model.subscriptions.Subscription
import ru.yandex.tours.storage.subscriptions.SubscriptionStorage
import ru.yandex.tours.subscriptions.RootSubscriptionActor
import ru.yandex.tours.util.Logging
import ru.yandex.vertis.scheduler.model.Schedule.{EveryHours, EveryMinutes}
import ru.yandex.vertis.scheduler.model.{Payload, Task, TaskDescriptor}

import scala.concurrent.ExecutionContext
import scala.util.{Failure, Success}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 10.08.15
 */
class NotifierScheduler(subscriptionStorage: SubscriptionStorage,
                        rootSubscriptionsActor: ActorRef)(implicit ec: ExecutionContext) extends Logging {

  def task = Task(
    TaskDescriptor("subscription_updater", EveryMinutes(5)),
    Payload.Async(() => {
      for (subscriptions <- subscriptionStorage.getAllSubscriptions) yield {
        rootSubscriptionsActor ! RootSubscriptionActor.AllSubscriptions(subscriptions)
      }
    })
  )

  private def isExpired(subscription: Subscription) = {
    subscription.subject.params.dates.when.end.isBefore(LocalDate.now)
  }

  def disableTask = Task(
    TaskDescriptor("subscription_disabler", EveryHours(1)),
    Payload.Async(() => {
      for (subscriptions <- subscriptionStorage.getAllSubscriptions) yield {
        for (subscription <- subscriptions if isExpired(subscription)) {
          subscriptionStorage.disableSubscription(subscription.user, subscription.id).onComplete {
            case Success(_) => log.info(s"Disabled expired subscription $subscription")
            case Failure(t) => log.error(s"Failed to disable subscription $subscription", t)
          }
        }
      }
    })
  )
}
