package ru.yandex.tours.subscriptions

import org.joda.time.{DateTime, LocalDate}
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.filter.snippet.HotelIdFilter
import ru.yandex.tours.model.filter.{HotelFilter, SnippetFilter}
import ru.yandex.tours.model.hotels.Hotel
import ru.yandex.tours.model.subscriptions.Notification.Payload
import ru.yandex.tours.model.subscriptions.{Notification, Subscription}
import ru.yandex.tours.operators.TourOperators
import ru.yandex.tours.prices.PriceSearcher
import ru.yandex.tours.util.Collections._
import ru.yandex.tours.util.Logging

import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, Future}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 10.08.15
 */
class DefaultNotificationBuilder(priceSearcher: PriceSearcher,
                                 hotelsIndex: HotelsIndex,
                                 tourOperators: TourOperators,
                                 tree: region.Tree)
                                (implicit ec: ExecutionContext) extends NotificationBuilder with Logging {

  private val FRESHNESS = 1.day
  private val HOTELS_IN_DIRECTION = 3
  private val OFFERS_IN_HOTEL = 3

  override def build(subscription: Subscription): Future[Notification] = {
    val payload: Future[Notification.Payload] = subscription.subject match {
      case Subscription.Direction(_, geoId, params) =>
        searchForDirection(geoId, params)
      case Subscription.Hotel(_, hotelId, params) =>
        hotelsIndex.getHotelById(hotelId) match {
          case Some(hotel) => searchForHotel(hotel, params)
          case None => Future.successful(Payload.empty)
        }
    }

    for (payload <- payload) yield {
      Notification(
        DateTime.now(),
        subscription,
        payload
      )
    }
  }

  private def searchForDirection(to: Int, params: Subscription.SubjectParams): Future[Notification.Payload] = {
    val snippetFilters = params.filters.collect { case snippetFilter: SnippetFilter => snippetFilter }
    val hotelFilters = params.filters.collect {
      case hotelFilter: HotelFilter => hotelFilter
    }.filterNot(_.isInstanceOf[HotelIdFilter])

    priceSearcher.searchLastPrices(params.from, to, params.ages, params.dates, snippetFilters, FRESHNESS)
      .map(_.filter(_.when.isAfter(LocalDate.now)))
      .map { prices =>
      val hotels = hotelsIndex.getHotels(prices.map(_.hotelId).distinct, hotelFilters, None)

      val offers = (for {
        p <- prices
        hotel <- hotels.get(p.hotelId)
        operator <- tourOperators.getById(p.operatorId)
      } yield {
        hotel -> Notification.Offer(p.when, p.nights, p.pansion, operator, p.price, p.created)
      }).toMultiMap.map {
        case (hotel, hotelOffers) => Notification.HotelOffers(hotel, hotelOffers)
      }.toSeq

      log.info(s"Got ${offers.size} offers to region $to for $params")
      if (offers.nonEmpty) {
        val cheapestOffer = offers.minBy(_.offers.map(_.price).min)
        val cheapestByStar = offers.groupBy(_.hotel.star).mapValues(_.minBy(_.offers.map(_.price).min)).toSeq

        val cheapestStar = cheapestOffer.hotel.star
        val filteredByStar = cheapestByStar.filter(_._1.id > cheapestStar.id).sortBy(_._1.id).map(_._2)

        val byRelevanceOffers = offers.toSeq.sortBy(_.hotel.relevance).reverse.take(HOTELS_IN_DIRECTION)
        val offersToShow = (cheapestOffer +: filteredByStar ++: byRelevanceOffers).distinct.take(HOTELS_IN_DIRECTION)
        Notification.DirectionOffers(offersToShow)
      } else {
        Notification.Payload.empty
      }
    }
  }

  private def searchForHotel(hotel: Hotel, params: Subscription.SubjectParams): Future[Notification.Payload] = {
    val snippetFilters = params.filters.collect {
      case snippetFilter: SnippetFilter => snippetFilter
    } :+ HotelIdFilter(Some(hotel.id))

    priceSearcher.searchLastPrices(params.from, hotel.geoId, params.ages, params.dates, snippetFilters, FRESHNESS)
      .map(_.filter(_.when.isAfter(LocalDate.now)))
      .map { prices =>
      val offers = for {
        p <- prices
        operator <- tourOperators.getById(p.operatorId)
      } yield {
        assert(p.hotelId == hotel.id, s"Got price from different hotel ${p.hotelId}. Expected ${hotel.id}")
        Notification.Offer(p.when, p.nights, p.pansion, operator, p.price, p.created)
      }

      val minOffers = offers.toSeq.sortBy(_.price).take(OFFERS_IN_HOTEL)
      log.info(s"Got ${offers.size} offers to hotel ${hotel.id} for $params")
      Notification.HotelOffers(hotel, minOffers)
    }.flatMap { hotelOffers =>
      if (hotelOffers.isEmpty) {
        val paramsWithoutHotel = params.copy(filters = params.filters.filterNot(_.isInstanceOf[HotelIdFilter]))
        tree.pathToRoot(hotel.geoId).foldLeft(Future.successful(Notification.Payload.empty)) {
          case (res, region) =>
            res.flatMap { result =>
              if (result.isEmpty) {
                searchForDirection(region.id, paramsWithoutHotel)
              }
              else Future.successful(result)
            }
        }
      }
      else Future.successful(hotelOffers)
    }
  }
}
