package ru.yandex.tours.subscriptions

import java.util.UUID

import akka.actor.{PoisonPill, Actor, Props}
import akka.serialization.Serialization
import org.apache.curator.framework.CuratorFramework
import ru.yandex.common.tokenization.{Ownerships, IntTokens, Owner, TokensDistributor}
import ru.yandex.tours.model.subscriptions.Subscription
import ru.yandex.tours.storage.subscriptions.NotificationStorage
import ru.yandex.tours.subscriptions.RootSubscriptionActor.{TokenSubscriptions, AllSubscriptions}
import ru.yandex.tours.subscriptions.render.Renderer
import ru.yandex.tours.subscriptions.sender.MailSender
import ru.yandex.tours.util.{Metrics, Logging}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 20.10.15
 */
class RootSubscriptionActor(zkClient: CuratorFramework,
                            config: TokensDistributor.Config,
                            subscriptionProps: Subscription => Props) extends Actor with Logging {

  val me = new Owner(UUID.randomUUID().toString, Map("path" -> Serialization.serializedActorPath(self)))
  val tokens = new IntTokens(64)
  var ownerships = Ownerships.empty(tokens)
  var buffer = Map.empty[String, Seq[Subscription]]

  val distributor = context.actorOf(Props(new TokensDistributor(me, tokens, zkClient, config)), "token-distributor")

  Metrics("actors." + self.path.name).newGauge("children") { context.children.size }

  @throws[Exception](classOf[Exception])
  override def preStart(): Unit = {
    super.preStart()
    distributor ! TokensDistributor.Subscribe
  }

  def buffer(token: String, subscriptions: Seq[Subscription]): Unit = {
    buffer += token -> subscriptions
  }

  def tryFlush(): Unit = {
    for ((token, subscriptions) <- buffer) {
      ownerships.getOwner(token) match {
        case Some(owner) =>
          log.info(s"Flushing ${subscriptions.size} subscriptions for token $token owned by $owner")
          sendTo(owner, TokenSubscriptions(token, subscriptions))
          buffer -= token
        case None =>
      }
    }
  }

  def sendTo(owner: Owner, tokenSubscriptions: TokenSubscriptions): Unit = {
    val path = owner.properties("path")
    context.actorSelection(path) ! tokenSubscriptions
  }

  override def receive: Receive = {
    case TokensDistributor.Command.Current(ownTokens, newOwnerships) =>
      log.debug(s"Current tokens received. " +
        s"It's ${ownTokens.mkString("[", ",", "]")}")
      ownerships = newOwnerships
      tryFlush()

    case TokensDistributor.Command.Take(token, newOwnerships) =>
      log.info(s"Take token <$token>.")
      ownerships = newOwnerships

    case TokensDistributor.Command.Return(token, newOwnerships) =>
      log.info(s"Return token <$token>.")
      context.children.filter(_ != distributor)
        .filter(ref => tokens.token(ref.path.name) == token)
        .foreach(_ ! PoisonPill)

      ownerships = newOwnerships

    case TokensDistributor.Command.ReturnAll =>
      log.info("Return all tokens.")
      context.children.filter(_ != distributor).foreach(_ ! PoisonPill)
      ownerships = ownerships.filterOwnerships(_.owner != me)

    case notification: TokensDistributor.Notification =>
      log.info("Got " + notification)

    case AllSubscriptions(allSubscriptions) =>
      val grouped = allSubscriptions.groupBy(s => tokens.token(s.id))
      tokens.foreach { token =>
        val subscriptions = grouped.getOrElse(token, Seq.empty)
        ownerships.getOwner(token) match {
          case Some(owner) =>
            sendTo(owner, TokenSubscriptions(token, subscriptions))
          case None =>
            log.info(s"No owner for token $token. Buffering ${subscriptions.size} subscriptions...")
            buffer(token, subscriptions)
        }
      }

    case TokenSubscriptions(token, subscriptions) if ownerships.getOwner(token).contains(me) =>
      val activeSubscriptions = subscriptions.filter(_.enabled)

      val ids = activeSubscriptions.map(_.id).toSet
      val current = context.children.filter(_ != distributor).map(_.path.name).filter(tokens.token(_) == token).toSet

      val toStop = current -- ids
      val toCreate = activeSubscriptions.filterNot(s => current.contains(s.id))

      toStop.foreach { id =>
        context.child(id).foreach(_ ! SubscriptionActor.Stop)
      }
      toCreate.foreach { subscription =>
        context.actorOf(subscriptionProps(subscription), subscription.id)
      }

    case msg @ TokenSubscriptions(token, subscriptions) =>
      ownerships.getOwner(token) match {
        case Some(owner) =>
          log.info(s"Got subscriptions for token $token owned by $owner. Redirecting...")
          sendTo(owner, msg)
        case None =>
          log.info(s"No owner for token $token. Buffering ${subscriptions.size} subscriptions...")
          buffer(token, subscriptions)
      }
  }
}

object RootSubscriptionActor {

  case class AllSubscriptions(subscriptions: Seq[Subscription])
  case class TokenSubscriptions(token: String, subscriptions: Seq[Subscription])

  def props(zkClient: => CuratorFramework, config: TokensDistributor.Config, subscriptionProps: Subscription => Props): Props = {
    Props(new RootSubscriptionActor(zkClient, config, subscriptionProps))
  }

  def props(zkClient: => CuratorFramework,
            config: TokensDistributor.Config,
            notificationStorage: NotificationStorage,
            notificationBuilder: NotificationBuilder,
            renderer: Renderer,
            mailSender: MailSender): Props = {
    props(zkClient, config, SubscriptionActor.props(_, notificationStorage, notificationBuilder, renderer, mailSender))
  }
}