package ru.yandex.tours.subscriptions

import akka.actor.{Actor, Cancellable, Props}
import org.joda.time.{DateTime, Duration}
import ru.yandex.tours.model.subscriptions.{Notification, Subscription}
import ru.yandex.tours.storage.subscriptions.NotificationStorage
import ru.yandex.tours.subscriptions.render.Renderer
import ru.yandex.tours.subscriptions.sender.MailSender
import ru.yandex.tours.util.Logging

import scala.concurrent.Future
import scala.concurrent.duration._
import scala.util.{Failure, Random, Success}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 09.09.15
 */
class SubscriptionActor(subscription: Subscription,
                        notificationStorage: NotificationStorage,
                        notificationBuilder: NotificationBuilder,
                        renderer: Renderer,
                        mailSender: MailSender) extends Actor with Logging {

  import SubscriptionActor._
  import context.dispatcher

  @throws[Exception](classOf[Exception])
  override def preStart(): Unit = {
    super.preStart()

    if (subscription.disabled) {
      stop()
    }
    notificationStorage.getLastNotification(subscription).onComplete {
      case Success(Some(notification)) =>
        scheduleCheck(from = notification.created, randomShiftBase = STARTUP_CHECK_DELAY)
      case Success(None) =>
        scheduleCheck(from = DateTime.now, delay = 1.second, randomShiftBase = STARTUP_CHECK_DELAY)
      case Failure(t) =>
        log.warn(s"Failed to load last notification for $subscription", t)
        scheduleCheck(from = DateTime.now, isFailure = true, randomShiftBase = STARTUP_CHECK_DELAY)
    }
  }

  def multiplyDelay(delay: FiniteDuration, m: Double): FiniteDuration = {
    require(!m.isInfinity, "m shouldNot be infinity")
    delay * m match {
      case fd: FiniteDuration => fd
      case d => sys.error(s"Expected FiniteDuration, got: $d")
    }
  }

  def scheduleCheck(from: DateTime,
                    delay: FiniteDuration = subscription.interval,
                    randomShiftBase: FiniteDuration = 5.minutes,
                    isFailure: Boolean = false): Cancellable = {
    val realDelay = if (isFailure) delay min FAILURE_RETRY_DELAY else delay
    val randomDelay = multiplyDelay(randomShiftBase, Random.nextDouble())
    val nextFireTime = from.plusMinutes(realDelay.toMinutes.toInt)
    val toSleep =
      (if (nextFireTime.isAfterNow) new Duration(DateTime.now, nextFireTime).getMillis
      else 0L) + randomDelay.toMillis
    log.info("Next check time: " + nextFireTime + " after " + toSleep.millis.toCoarsest + " for " + subscription)
    context.system.scheduler.scheduleOnce(toSleep.millis, self, Check)
  }

  def shouldSend(notification: Notification, lastNotification: Option[Notification]): Boolean = {
    if (notification.payload.isEmpty) {
      log.info(s"Skipping notification ($subscription): nothing to send")
      return false
    }
    if (notification.payload.nonEmpty && lastNotification.isEmpty) return true
    if (lastNotification.nonEmpty) {
      val last = lastNotification.get
      def isElapsed(interval: FiniteDuration) = {
        notification.created.getMillis - last.created.getMillis >= interval.toMillis
      }

      if (!isElapsed(subscription.interval)) {
        log.info(s"Skipping notification ($subscription): min interval not elapsed")
        return false
      }

      if (notification.payload.minPrice == last.payload.minPrice && !isElapsed(subscription.interval * 3)) {
        log.info(s"Skipping notification ($subscription): min price not changed and three intervals not elapsed")
        return false
      }
    }
    true
  }

  def stop(): Unit = {
    log.info(s"Subscription $subscription is disabled. Stopping")
    context.stop(self)
  }

  def sendNotification(notification: Notification): Future[Boolean] = {
    notificationStorage.getLastNotification(subscription).flatMap { lastNotification =>
      if (shouldSend(notification, lastNotification)) {
        for {
          letter <- renderer.render(notification, lastNotification)
          _ = log.info(s"Sending letter [${letter.content.subject}] to [${letter.recipient}] for $subscription")
          _ = mailSender.send(letter)
          _ <- notificationStorage.saveNotification(notification)
        } yield {
          SubscriptionsMetrics.report(notification)
          true
        }
      } else {
        Future.successful(false)
      }
    }
  }

  override def receive: Receive = {
    case Check =>
      notificationBuilder.build(subscription).onComplete {
        case Success(notification) =>
          self ! Send(notification)
        case Failure(t) =>
          log.warn(s"Failed to build notification for $subscription", t)
          scheduleCheck(from = DateTime.now, isFailure = true)
      }

    case Send(notification) =>
      sendNotification(notification).onComplete {
        case Success(true) =>
          scheduleCheck(from = notification.created)
        case Success(false) =>
          scheduleCheck(from = notification.created, isFailure = true)
        case Failure(t) =>
          log.error(s"Failed to send notification for $subscription", t)
          scheduleCheck(from = notification.created, isFailure = true)
      }
    case Stop =>
      stop()
  }
}

object SubscriptionActor {
  private case object Check
  private case class Send(notification: Notification)
  case object Stop

  private val FAILURE_RETRY_DELAY = 1.hour
  private val STARTUP_CHECK_DELAY = 1.hour

  def props(subscription: Subscription, notificationStorage: NotificationStorage,
            notificationBuilder: NotificationBuilder, renderer: Renderer, mailSender: MailSender): Props = {
    Props(new SubscriptionActor(subscription, notificationStorage, notificationBuilder, renderer, mailSender))
  }
}