package ru.yandex.tours.subscriptions

import java.util.concurrent.TimeUnit

import ru.yandex.tours.model.subscriptions.Notification
import ru.yandex.tours.util.{Logging, Metrics}

import scala.concurrent.duration.FiniteDuration

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 06.11.15
 */
object SubscriptionsMetrics extends Logging {
  private val metrics = Metrics("notifications")
  private val priceFreshness = metrics.getHistogram("price_freshness")
  private val sentMeter = metrics.getMeter("sent")

  def report(notification: Notification): Unit = {
    val allOffers = notification.payload match {
      case Notification.DirectionOffers(hotelOffers) => hotelOffers.map(_.offers.minBy(_.price))
      case Notification.HotelOffers(_, offers) => offers.sortBy(_.price).take(3)
      case _ => Seq.empty
    }
    for (offer <- allOffers) {
      val delayInMillis = notification.created.getMillis - offer.created.getMillis
      val delay = FiniteDuration(delayInMillis, TimeUnit.MILLISECONDS)
      priceFreshness.update(delay.toMinutes)
    }
    val oldOfferWaterline = notification.created.minusHours(1)
    val oldOffers = allOffers.filter(_.created.isBefore(oldOfferWaterline))
    if (oldOffers.nonEmpty) {
      log.info(s"Sent old ${oldOffers.size} offers for subscriptions ${notification.subscription.id}: $oldOffers")
    }

    sentMeter.mark()
  }
}
