package ru.yandex.tours.subscriptions.render

import akka.util.Timeout
import com.google.common.base.Charsets
import org.joda.time.format.DateTimeFormat
import org.json.{JSONArray, JSONObject}
import ru.yandex.tours.direction.Directions
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.{HotelsIndex, HotelsVideo}
import ru.yandex.tours.model.Languages
import ru.yandex.tours.model.hotels.Hotel
import ru.yandex.tours.model.subscriptions.Notification.{HotelOffers, DirectionOffers}
import ru.yandex.tours.model.subscriptions.Subscription.Direction
import ru.yandex.tours.model.subscriptions._
import ru.yandex.tours.model.util.{CustomNights, FlexNights, CustomDateInterval, FlexDateInterval}
import ru.yandex.tours.model.utm.UtmMark
import ru.yandex.tours.search.settings.SearchSettingsHolder
import ru.yandex.tours.serialize.UrlBuilder
import ru.yandex.tours.serialize.json.CommonJsonSerialization
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.lang.Dates._
import spray.http.StatusCodes

import scala.collection.JavaConverters._
import scala.concurrent.{ExecutionContext, Future}
import scala.concurrent.duration._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 27.08.15
 */
class HttpRenderer(httpClient: AsyncHttpClient,
                   host: String, port: Int,
                   hotelsIndex: HotelsIndex,
                   hotelsVideo: HotelsVideo,
                   geoMappingHolder: GeoMappingHolder,
                   searchSettingsHolder: SearchSettingsHolder,
                   tree: region.Tree,
                   directions: Directions,
                   urlBuilder: UrlBuilder)
                  (implicit ec: ExecutionContext) extends Renderer with CommonJsonSerialization with Logging {

  private val uri = s"http://$host:$port/tours/sub"
  private val lang = Languages.ru
  private val format = DateTimeFormat.forPattern("yyyy-MM-dd")

  private implicit val timeout = Timeout(10.seconds)
  private val utm = UtmMark.empty.withSource("subscription").withCampaign("list_unsubscribe").withTerm("unsubscribe")

  private def messageIdFor(notification: Notification) =
    "<subscription." + notification.subscription.id + "." + notification.created.getMillis + "@travel.yandex.ru>"

  private def unsubscribeLink(subscription: Subscription) =
    s"https://${urlBuilder.domain}/subscriptions/${subscription.id}?act=remove&${subscription.user.identityType}=${subscription.user.identity}&${utm.urlSafe}"

  private def headersFor(notification: Notification, previous: Option[Notification]) = {
    Seq(
      "List-Unsubscribe" -> ("<" + unsubscribeLink(notification.subscription) + ">")
    ) ++ previous.fold(Seq.empty[(String, String)])(n => Seq("In-Reply-To" -> messageIdFor(n)))
  }

  private def toRegion(regionId: Int): String = tree.region(regionId).fold("") {
    r => r.preposition + " " + r.accusative
  }

  private def toHotel(hotel: Hotel): String = "в " + hotel.name(lang)

  private def user(user: UserIdentity) = {
    new JSONObject()
      .put(user.identityType, user.identity)
  }

  override def render(notification: Notification, previous: Option[Notification]): Future[Letter] = {
    notification.subscription.subject match {
      case _: Subscription.Direction => renderDirection(notification, previous)
      case _: Subscription.Hotel if notification.payload.isInstanceOf[Notification.DirectionOffers] => renderFallback(notification, previous)
      case _: Subscription.Hotel => renderHotel(notification, previous)
    }
  }

  private val fromAddress = Address("travel-subscriptions@yandex-team.ru", Some("Яндекс.Путешествия"))

  def renderDirection(notification: Notification, previous: Option[Notification]): Future[Letter] = {
    val subject = notification.subscription.subject.asInstanceOf[Direction]

    var title = "✈ Туры " + toRegion(subject.geoId)
    val ages = subject.params.ages
    val region = tree.region(subject.geoId).getOrElse(sys.error(s"Unknown region: ${subject.geoId}"))
    val direction = directions.get(region.id)

    val json = new JSONObject()
    val sub = new JSONObject()
      .put("query", subject.query)
      .put("from", subject.params.from)
      .put("to", direction.fold(regionToJson(region, lang))(directionToJson(_, lang)))
      .put("type", "direction")
    json.put("subscription", sub)
    json.put("subscription-id", notification.subscription.id)
    json.put("user", user(notification.subscription.user))

    json.put("view", new JSONObject().put("tld", "ru"))

    val payload = notification.payload.asInstanceOf[DirectionOffers]

    if (payload.nonEmpty) {
      val minPrice = payload.minPrice
      val prevMinPrice = previous
        .filter(_.payload.isInstanceOf[Notification.DirectionOffers])
        .filter(_.payload.nonEmpty)
        .map(_.payload.minPrice)
      val change = prevMinPrice.map { prev =>
        val diff = minPrice - prev
        if (diff > 0) s" (+${diff}р.)"
        else if (diff < 0) s" (${diff}р.)"
        else " (Цена не изменилась)"
      }

      title += " от " + minPrice + "р." + change.getOrElse("")
    }

    val offersArray = new JSONArray()
    for (HotelOffers(hotel, offers) <- payload.offers) {
      val o = new JSONObject()
      o.put("hotel", toJson(hotel, hotelsVideo, tree, geoMappingHolder, searchSettingsHolder, lang))

      o.put("price_from", offers.map(_.price).min)
      o.put("price_to", offers.map(_.price).max)
      o.put("start_date_from", offers.map(_.when).min)
      o.put("start_date_to", offers.map(_.when).max)
      o.put("nights_from", offers.map(_.nights).min)
      o.put("nights_to", offers.map(_.nights).max)

      o.put("pansions", new JSONArray(offers.map(_.pansion.toString).distinct.asJava))
      o.put("ages", new JSONArray(ages.asJava))
      o.put("offer_count", offers.size)
      o.put("tour_operators", new JSONArray(offers.map(_.operator).distinct.map {
        operator => new JSONObject().put("operator_name", operator.name).put("operator_id", operator.id)
      }.asJava))

      val minOffer = offers.minBy(_.price)

      val req = filtersToJson(subject.params.filters)
        .put("from", subject.params.from)
        .put("to", regionToJson(region, lang).put("hotel_id", hotel.id))
        .put("when", minOffer.when.toString(format))
        .put("nights", minOffer.nights)
        .put("ages", new JSONArray(ages.asJava))
        .put("when_flex", false)
        .put("nights_flex", false)

      o.put("request", req)

      offersArray.put(o)
    }
    json.put("documents", new JSONObject().put("list", offersArray).put("all_offers_request", allOffersRequest(subject)))

    httpClient.post(uri, json.toString.getBytes(Charsets.UTF_8), List("Content-Type" -> "application/json")).map {
      case (StatusCodes.OK, html) =>
        val email = notification.subscription.email

        val id = messageIdFor(notification)
        val headers = headersFor(notification, previous)

        Letter(
          LetterContent(title, Body.Html(html)),
          fromAddress,
          Recipient(Address(email)),
          messageId = Some(id),
          headers = Headers(headers)
        )
      case (status, body) => sys.error(s"Failed to render letter. Code = $status, body = $body")
    }
  }
  def renderFallback(notification: Notification, previous: Option[Notification]): Future[Letter] = {
    val subject = notification.subscription.subject.asInstanceOf[Subscription.Hotel]
    val payload = notification.payload.asInstanceOf[DirectionOffers]

    val hotel = hotelsIndex.getHotelById(subject.hotelId).getOrElse(sys.error(s"Unknown hotel ${subject.hotelId}"))
    val title = "✈ Туры " + toHotel(hotel)
    val ages = subject.params.ages
    val region = tree.region(hotel.geoId).getOrElse(sys.error(s"Unknown region: ${hotel.geoId}"))
    val direction = directions.get(hotel.geoId)

    val json = new JSONObject()
    val sub = new JSONObject()
      .put("query", subject.query)
      .put("from", subject.params.from)
      .put("to", direction.fold(regionToJson(region, lang))(directionToJson(_, lang)))
      .put("hotel", toJson(hotel, hotelsVideo, tree, geoMappingHolder, searchSettingsHolder, lang))
      .put("type", "hotel")
      .put("fallback", true)
    json.put("subscription", sub)
    json.put("subscription-id", notification.subscription.id)
    json.put("user", user(notification.subscription.user))

    json.put("view", new JSONObject().put("tld", "ru"))

    val allOffers = payload.offers

    val offersArray = new JSONArray()
    for (HotelOffers(hotel, offers) <- allOffers) {
      val o = new JSONObject()
      o.put("hotel", toJson(hotel, hotelsVideo, tree, geoMappingHolder, searchSettingsHolder, lang))

      o.put("price_from", offers.map(_.price).min)
      o.put("price_to", offers.map(_.price).max)
      o.put("start_date_from", offers.map(_.when).min)
      o.put("start_date_to", offers.map(_.when).max)
      o.put("nights_from", offers.map(_.nights).min)
      o.put("nights_to", offers.map(_.nights).max)

      o.put("pansions", new JSONArray(offers.map(_.pansion.toString).distinct.asJava))
      o.put("ages", new JSONArray(ages.asJava))
      o.put("offer_count", offers.size)
      o.put("tour_operators", new JSONArray(offers.map(_.operator).distinct.map {
        operator => new JSONObject().put("operator_name", operator.name).put("operator_id", operator.id)
      }.asJava))

      val minOffer = offers.minBy(_.price)

      val req = filtersToJson(subject.params.filters)
        .put("from", subject.params.from)
        .put("to", regionToJson(region, lang).put("hotel_id", hotel.id))
        .put("when", minOffer.when.toString(format))
        .put("nights", minOffer.nights)
        .put("ages", new JSONArray(ages.asJava))
        .put("when_flex", false)
        .put("nights_flex", false)

      o.put("request", req)

      offersArray.put(o)
    }
    json.put("documents", new JSONObject().put("list", offersArray).put("all_offers_request", allOffersRequest(subject)))

    httpClient.post(uri, json.toString.getBytes(Charsets.UTF_8), List("Content-Type" -> "application/json")).map {
      case (StatusCodes.OK, html) =>
        val email = notification.subscription.email

        val id = messageIdFor(notification)
        val headers = headersFor(notification, previous)

        Letter(
          LetterContent(title, Body.Html(html)),
          fromAddress,
          Recipient(Address(email)),
          messageId = Some(id),
          headers = Headers(headers)
        )
      case (status, body) => sys.error(s"Failed to render letter. Code = $status, body = $body")
    }
  }

  def renderHotel(notification: Notification, previous: Option[Notification]): Future[Letter] = {
    val subject = notification.subscription.subject.asInstanceOf[Subscription.Hotel]
    val payload = notification.payload.asInstanceOf[HotelOffers]

    val hotel = payload.hotel
    var title = "✈ Туры " + toHotel(hotel)
    val ages = subject.params.ages
    val region = tree.region(hotel.geoId).getOrElse(sys.error(s"Unknown region: ${hotel.geoId}"))
    val direction = directions.get(hotel.geoId)

    val json = new JSONObject()
    val sub = new JSONObject()
      .put("query", subject.query)
      .put("from", subject.params.from)
      .put("to", direction.fold(regionToJson(region, lang))(directionToJson(_, lang)))
      .put("hotel", toJson(hotel, hotelsVideo, tree, geoMappingHolder, searchSettingsHolder, lang))
      .put("type", "hotel")
    json.put("subscription", sub)
    json.put("subscription-id", notification.subscription.id)
    json.put("user", user(notification.subscription.user))

    json.put("view", new JSONObject().put("tld", "ru"))

    val allOffers = payload.offers

    if (allOffers.nonEmpty) {
      val minPrice = allOffers.map(_.price).min
      val prevMinPrice = previous
        .filter(_.payload.isInstanceOf[Notification.HotelOffers])
        .map(_.payload.asInstanceOf[Notification.HotelOffers].offers)
        .filter(_.nonEmpty)
        .map(_.map(_.price).min)
      val change = prevMinPrice.map { prev =>
        val diff = minPrice - prev
        if (diff > 0) s" (+${diff}р.)"
        else if (diff < 0) s" (${diff}р.)"
        else " (Цена не изменилась)"
      }

      title += " от " + minPrice + "р." + change.getOrElse("")
    }

    val offersArray = new JSONArray()
    for (offer <- allOffers) {
      val o = new JSONObject()

      o.put("price", offer.price)
      o.put("start_date", offer.when)
      o.put("nights", offer.nights)

      o.put("pansion", offer.pansion.toString)
      o.put("ages", new JSONArray(ages.asJava))
      o.put("tour_operator", new JSONObject().put("operator_name", offer.operator.name).put("operator_id", offer.operator.id))


      val req = filtersToJson(subject.params.filters)
        .put("from", subject.params.from)
        .put("to", regionToJson(region, lang).put("hotel_id", hotel.id))
        .put("when", offer.when.toString(format))
        .put("nights", offer.nights)
        .put("ages", new JSONArray(ages.asJava))
        .put("when_flex", false)
        .put("nights_flex", false)

      o.put("request", req)

      offersArray.put(o)
    }
    json.put("documents", new JSONObject().put("list", offersArray).put("all_offers_request", allOffersRequest(subject)))

    httpClient.post(uri, json.toString.getBytes(Charsets.UTF_8), List("Content-Type" -> "application/json")).map {
      case (StatusCodes.OK, html) =>
        val email = notification.subscription.email

        val id = messageIdFor(notification)
        val headers = headersFor(notification, previous)

        Letter(
          LetterContent(title, Body.Html(html)),
          fromAddress,
          Recipient(Address(email)),
          messageId = Some(id),
          headers = Headers(headers)
        )
      case (status, body) => sys.error(s"Failed to render letter. Code = $status, body = $body")
    }
  }

  private def allOffersRequest(subject: Subscription.Subject) = {
    val params = subject.params
    val json = filtersToJson(params.filters)
    json.put("from", params.from)
    json.put("ages", params.ages.asJava)

    params.dates.when match {
      case FlexDateInterval(when, flex) => json.put("when", when.toString).put("when_flex", flex)
      case CustomDateInterval(start, end) => json.put("when", start.toString)
    }
    params.dates.nights match {
      case FlexNights(nights, flex) => json.put("nights", nights).put("nights_flex", flex)
      case CustomNights(nights) => json.put("nights", nights.head)
    }
    subject match {
      case Subscription.Direction(_, geoId, _) => json.put("to", geoId)
      case Subscription.Hotel(_, hotelId, _) => json.put("hotel_id", hotelId)
    }

    json
  }
}
