package ru.yandex.tours.subscriptions.render

import ru.yandex.tours.geo.base.{Region, region}
import ru.yandex.tours.model.Languages
import ru.yandex.tours.model.subscriptions.Notification.HotelOffers
import ru.yandex.tours.model.subscriptions._
import ru.yandex.tours.tanker.Translations
import ru.yandex.tours.util.Collections._
import ru.yandex.tours.util.lang.Dates._

import scala.concurrent.Future

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 07.08.15
 */
class SimpleRenderer(tree: region.Tree, translations: Translations) extends Renderer {

  private def nameFor(regionId: Int): String = tree.region(regionId).fold("")(nameFor)
  private def nameFor(region: Region): String = region.name(Languages.ru)
  private def toRegion(regionId: Int): String = tree.region(regionId).fold("") {
    r => r.preposition + " " + r.accusative
  }

  private val lang = Languages.ru

  private def messageIdFor(notification: Notification) =
    "<subscription." + notification.subscription.id + "." + notification.created.getMillis + "@travel.yandex.ru>"

  override def render(notification: Notification, previous: Option[Notification]): Future[Letter] = {
    val subscription = notification.subscription
    val subject = subscription.subject.asInstanceOf[Subscription.Direction]
    val payload = notification.payload.asInstanceOf[Notification.DirectionOffers]

    var title = "Туры " + toRegion(subject.geoId)
    val text = new StringBuilder

    text ++= "Здравствуйте!\n"

    if (payload.offers.isEmpty) {
      text ++= "По вашим интересам ничего не найдено.\n\n"
    } else {
      val minPrice = payload.offers.flatMap(_.offers).map(_.price).min
      val prevMinPrice = previous
        .filter(_.payload.isInstanceOf[Notification.DirectionOffers])
        .map(_.payload.asInstanceOf[Notification.DirectionOffers].offers)
        .filter(_.nonEmpty)
        .map(_.flatMap(_.offers).map(_.price).min)
      val change = prevMinPrice.map { prev =>
        val diff = minPrice - prev
        if (diff > 0) s" (+${diff}р.)"
        else if (diff < 0) s" (${diff}р.)"
        else " (Цена не изменилась)"
      }

      title += " от " + minPrice + "р." + change.getOrElse("")

      text ++= "Мы собрали интересные предложения "
      text ++= toRegion(subject.geoId)
      text ++= " по вашим предпочтениям.\n\n"

      for (HotelOffers(hotel, offers) <- payload.offers) {
        text ++= hotel.name(lang)
        text += ' '
        text append hotel.star.id
        text += '*'
        text += '\n'

        text ++= nameFor(hotel.geoId)
        text += '\n'

        text append hotel.reviewsCount
        text ++= " отзывов"
        text += '\n'

        val (minWhen, maxWhen) = offers.map(_.when).minMax
        val (minNights, maxNights) = offers.map(_.nights).minMax
        text ++= "Вылеты "
        text ++= minWhen.toString
        text ++= " – "
        text ++= maxWhen.toString
        text ++= ", на "
        text ++= minNights.toString
        text ++= "–"
        text ++= maxNights.toString
        text ++= " ночей"
        text += '\n'

        text ++= "От "
        text append offers.map(_.price).min
        text ++= " рублей на "
        text append subject.params.ages.size
        text ++= " человек"

        text ++= "\n\n"
      }
    }

    val email = notification.subscription.email

    val id = messageIdFor(notification)
    val headers = previous.fold(Seq.empty[(String, String)])(n => Seq("In-Reply-To" -> messageIdFor(n)))

    Future.successful {
      Letter(
        LetterContent(title, Body.Text(text.toString())),
        Address("tourism-dev@yandex-team.ru", Some("Яндекс.Путешествия")),
        Recipient(Address(email)),
        messageId = Some(id),
        headers = Headers(headers)
      )
    }
  }
}
