package ru.yandex.tours.subscriptions.sender

import java.net.IDN
import java.util.{Date, Properties}
import javax.mail.internet.{InternetAddress, MimeMessage}
import javax.mail.{Message, Session, Transport}

import org.apache.commons.lang.StringUtils
import ru.yandex.tours.model.subscriptions.{Address, Body, Letter}

import scala.concurrent.duration._
import scala.language.implicitConversions

/**
 * [[MailSender]] implementation with aim of JavaMail library.
 * It works via SMTP protocol through SMTP relay.
 *
 * @param smtpHost SMTP host or relay name
 * @param smtpPort SMPT port, 25 by default
 * @param connectionTimeout timeout to connect to SMTP host
 * @param debug if true JavaMail will output debug messages to log
 */
class JavaMailSender(smtpHost: String,
                     smtpPort: Int = JavaMailSender.DefaultSmtpPort,
                     connectionTimeout: FiniteDuration = JavaMailSender.ConnectionTimeout,
                     debug: Boolean = false) extends MailSender {

  import JavaMailSender._

  private val properties = new Properties()
  properties.put("mail.smtp.host", smtpHost)
  properties.put("mail.smtp.port", smtpPort.toString)
  properties.put("mail.smtp.connectiontimeout", connectionTimeout.toMillis.toString)
  properties.put("mail.debug", debug.toString)

  private val session = Session.getDefaultInstance(properties)

  override def send(letter: Letter): Unit = {
    if (smtpHost.isEmpty) {
      return
    }
    val message = new MimeMessage(session) {
      override def updateMessageID(): Unit = {
        letter.messageId match {
          case Some(messageId) ⇒ setHeader("Message-ID", messageId)
          case None ⇒ super.updateMessageID()
        }
      }
    }
    message.addFrom(Array(letter.sender))
    message.addRecipient(Message.RecipientType.TO, escapeDomain(letter.recipient.address))
    for (cc ← letter.recipient.other) {
      message.addRecipient(Message.RecipientType.CC, escapeDomain(cc))
    }
    message.setSubject(letter.content.subject, utf8)

    letter.content.body match {
      case Body.Text(value) ⇒
        message.setText(value, utf8)
      case Body.Html(value) ⇒
        message.setContent(value, HtmlType)
        message.setHeader("Content-Transfer-Encoding", "base64")
    }

    for ((name, value) ← letter.headers.list)
      message.addHeader(name, value)

    message.setSentDate(new Date())
    message.saveChanges()

    Transport.send(message)
  }

  override val toString = s"JavaMailer(smptHost=$smtpHost, smptPort=$smtpPort)"
}

object JavaMailSender {

  val ConnectionTimeout = 3.seconds

  val DefaultSmtpPort = 25

  val utf8 = "utf-8"

  val HtmlType = "text/html; charset=utf-8"

  def escapeDomain(a: Address): Address = {
    val Array(addr, domain) = StringUtils.splitPreserveAllTokens(a.value, '@')
    val newEmail = addr + "@" + IDN.toASCII(domain)
    a.copy(value = newEmail)
  }

  implicit def toInternetAddress(a: Address): javax.mail.Address =
    new InternetAddress(a.value, a.view.orNull, utf8)
}
