package ru.yandex.tours

import java.util.UUID
import java.util.concurrent.ConcurrentHashMap

import org.joda.time.LocalDate
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.BaseModel.Pansion
import ru.yandex.tours.model.RequestStatus.RequestStatus
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.{RequestStatus, TourOperator}
import ru.yandex.tours.operators.TourOperators
import ru.yandex.tours.util.spray.HttpHandler
import spray.http.{MediaTypes, StatusCode, StatusCodes}
import spray.routing.{RequestContext, Route}

import scala.collection.JavaConversions._
import scala.concurrent.duration._
import scala.util.Random

class LtHandler(hotelsIndex: HotelsIndex, tourOperators: TourOperators) extends HttpHandler {
  val hotelIds = hotelsIndex.hotels.flatMap(_.partnerIds.find(_.partner == Partners.lt)).map(_.id.toInt).toSet
  val cacheTTL = 20.minutes
  val MAX_HOTEL_COUNT = 1000
  val statuses = new ConcurrentHashMap[String, (Map[TourOperator, RequestStatus], Long)]
  val failProbability = 0.1

  override def route: Route = {
    dynamic {
      pathPrefix("search") {
        path("enqueue") {
          val requestId = UUID.randomUUID().toString
          val operatorCount = Random.nextInt(4) + 7
          val operators = Random.shuffle(tourOperators.getAll).take(operatorCount).map(_ -> RequestStatus.PENDING).toMap
          statuses.put(requestId, (operators, System.currentTimeMillis()))
          scheduleUpdate(requestId)
          val response = enqueueResponse(requestId, operators)
          completeJson(response)
        } ~ (path("status") & parameter('request_id)) { requestId =>
          withOperators(requestId) { operators =>
            completeJson(s"{${statusResponse(requestId, operators)}}")
          }
        } ~ (path("result_compact") & parameters('request_id, 'operator_ids ? "")) { (requestId, requestedOperatorsRaw) =>
          withOperators(requestId) { operators =>
            var requestedOperators: Iterable[Int] = requestedOperatorsRaw.split(",").map(_.trim).filter(_.nonEmpty).map(_.toInt)
            if (requestedOperators.isEmpty) {
              requestedOperators = operators.keys.map(_.id)
            }
            val hotels = Random.shuffle(hotelIds).take(Random.nextInt(MAX_HOTEL_COUNT) + 1)
            val response = hotels.map(id => randomSnippet(id, requestedOperators)).mkString(",")
            completeJson(s"[$response]")
          }
        }
      }
    }
  }

  private def scheduleUpdate(requestId: String) {
      context.system.scheduler.scheduleOnce(getUpdateTime, new Runnable {
        override def run(): Unit = updateStatus(requestId)
      })
  }

  private def updateStatus(requestId: String) = {
    val (map, timestamp) = statuses.get(requestId)
    val uncompleted = map.filter(t => !RequestStatus.isFinished(t._2))
    var schedule = false
    val updated = map ++ (if (uncompleted.size < 3) {
      uncompleted
    } else {
      schedule = true
      uncompleted.take(uncompleted.size / 2)
    }).map(t => t._1 -> getFinishedStatus)
    statuses.put(requestId, (updated, timestamp))
    if (schedule) scheduleUpdate(requestId)
  }

  private def getUpdateTime = (Random.nextInt(3) + 1).seconds

  private def getFinishedStatus = if (Random.nextDouble() < failProbability) RequestStatus.FAILED else RequestStatus.COMPLETED
  
  private def randomSnippet(hotelId: Int, operators: Iterable[Int]) = {
    val minPrice = Random.nextInt(100000)
    val diff = Random.nextInt(100000) + 1000
    val maxPrice = minPrice + diff
    val offerCount = Random.nextInt(100)
    val operator = Random.shuffle(operators).head
    val nights = Random.nextInt(15) + 1
    val startDate = LocalDate.now().plusDays(Random.nextInt(30)).toString
    val pansions = Random.shuffle(Pansion.values().toIterable).take(Random.nextInt(Pansion.values.size) + 1).map(pansion => {
      s""""$pansion":${minPrice + Random.nextInt(diff)}"""
    }).mkString(", ")
    s"""
      |{
      |"hotel_id" : $hotelId,
      |"min_price" : $minPrice,
      |"max_price" : $maxPrice,
      |"offer_count" : $offerCount,
      |"operator" : $operator,
      |"pansions": {$pansions},
      |"nights" : $nights,
      |"start_date" : "$startDate"
      |}
    """.stripMargin
  }

  private def withOperators(requestId: String)(handle: Map[TourOperator, RequestStatus] => RequestContext => Unit) = {
    Option(statuses.get(requestId)) match {
      case Some((operators, timestamp)) => handle(operators)
      case None => completeJson("""{"error":"unknown requestId"}""", StatusCodes.BadRequest)
    }
  }

  private def completeJson(response: String, statusCode: StatusCode = StatusCodes.OK): RequestContext => Unit = {
    respondWithMediaType(MediaTypes.`application/json`) {
      complete(statusCode, response)
    }
  }

  private def statusResponse(requestId: String, operators: Map[TourOperator, RequestStatus]) = {
    val statuses = operators.map({case (operator, status) => s""""${operator.id}": "${status.toString}""""})
    s"""
      |"partner": "Ya",
      |"request_id": "$requestId",
      |"status": {
      |${statuses.mkString(",\n")}
      |}
    """.stripMargin
  }

  context.system.scheduler.schedule(1.minute, 1.minute, new Runnable {
    override def run(): Unit = {
      statuses.entrySet().foreach(t => if (System.currentTimeMillis() - t.getValue._2 > cacheTTL.toMillis) statuses.remove(t.getKey))
    }
  })

  private def enqueueResponse(requestId: String, operators: Map[TourOperator, RequestStatus]) = {
    s"""{
      |${statusResponse(requestId, operators)}
      |,
      |"search_params":
      |{
      |
      |    "query": "Moscow-RU-to-Any-EG-departure-25.03.2015-for-7-nights-2-adults-0-kids-1..5-stars",
      |    "from":
      |
      |{
      |
      |    "city":
      |
      |{
      |
      |    "id": 2,
      |    "name_ru": "Москва",
      |    "name_en": "Moscow"
      |
      |},
      |"country":
      |
      |    {
      |        "id": 2,
      |        "name_ru": "Россия",
      |        "name_en": "Russia",
      |        "iso2": "RU"
      |    }
      |
      |},
      |"to":
      |{
      |
      |    "city":
      |
      |{
      |
      |    "id": 0,
      |    "name_ru": "Любой город",
      |    "name_en": "Any"
      |
      |},
      |"country":
      |{
      |
      |    "id": 3,
      |    "name_ru": "Египет",
      |    "name_en": "Egypt",
      |    "iso2": "EG",
      |    "wb_code": "800260"
      |
      |},
      |"hotel_ids":
      |
      |    [
      |        40069
      |    ]
      |
      |},
      |"start_date": "2015-03-25",
      |"nights":
      |{
      |
      |    "from": 7,
      |    "to": 7
      |
      |},
      |"adults": 2,
      |"kids":
      |{
      |
      |    "count": 0,
      |    "ages": [ ]
      |
      |},
      |"stars":
      |{
      |
      |    "from": 1,
      |    "to": 5
      |
      |},
      |"options":
      |
      |        {
      |            "from_price": 0,
      |            "wide_search": false,
      |            "flex_dates": false
      |        }
      |    }
      |
      |}""".stripMargin
  }
}
