package ru.yandex.tours

import akka.actor.Props
import org.apache.cxf.transport.common.gzip.GZIPFeature
import ru.yandex.tours.app._
import ru.yandex.tours.backend.events.HydraLogger
import ru.yandex.tours.backend.events.PriceHistoryTransfer
import ru.yandex.tours.backend.recommend.{DirectionRecommendationService, LocalRecommendService}
import ru.yandex.tours.backend.search.{Searcher2Adapter, LocalHotelSearchService, LocalOffersSearchService, SearcherActor}
import ru.yandex.tours.backend.transfer.TransferSearchSupport
import ru.yandex.tours.billing.BillingComponents
import ru.yandex.tours.clickhouse.ClickHouseSupport
import ru.yandex.tours.extdata.ExtDataSupport
import ru.yandex.tours.hotels.CMHotelsBillingIndex
import ru.yandex.tours.model.Source
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.search.SearchType
import ru.yandex.tours.operators.{HotelProvidersAvailability, SearchSources, TourOperatorAvailability}
import ru.yandex.tours.partners._
import ru.yandex.tours.partners.booking.{BookingResponseParser, DefaultBookingClient}
import ru.yandex.tours.partners.common.{CommonHotelProviderClient, CommonTourOperatorClient, CommonTourOperatorResponseParser}
import ru.yandex.tours.partners.hotelscombined.HotelsCombinedSearcher
import ru.yandex.tours.partners.hotelscombined.client.{DefaultHotelsCombinedClient, InstrumentedHotelsCombinedClient, ThrottledHotelsCombinedClient}
import ru.yandex.tours.partners.leveltravel.parsers.ActualizationParser
import ru.yandex.tours.partners.leveltravel.{DefaultLtClient, InstrumentedLtClient, LevelTravelSearcher, ThrottledLtClient}
import ru.yandex.tours.partners.oktogo.{DefaultOktogoClient, OktogoTranslator, TravelApiService}
import ru.yandex.tours.partners.ostrovokv3.{DefaultOstrovokV3Client, OstrovokV3ResponseParser}
import ru.yandex.tours.partners.sunmar._
import ru.yandex.tours.personalization.DummyDirectionInterestService
import ru.yandex.tours.prices.PriceHistoryStorage
import ru.yandex.tours.searcher.api.{RootSearcherHttpHandler, SearcherHandler}
import ru.yandex.tours.storage.StorageComponents
import ru.yandex.tours.storage.minprice.LocalMinPriceService
import ru.yandex.tours.util.spray.{PingControl, RouteesContext}
import ru.yandex.tours.util.zoo.{BooleanSerializer, SharedValue}
import ru.yandex.tours.util.{LoggingFeature, Throttler, WsdlUtils}
import shapeless.HNil

import scala.concurrent.duration._
import scala.reflect.ClassTag
import scala.util.Try

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 22.01.15
 */
trait SearcherComponents {
  this: Application
    with AkkaSupport with SpraySupport
    with StorageComponents
    with CommonDataHolders
    with ExtDataSupport
    with HttpSupport
    with ClickHouseSupport
    with ZookeeperSupport
    with AviaSearcherSupport
    with TransferSearchSupport
    with BillingComponents =>

  import akkaSystem.dispatcher

  lazy val searcher2adapter = {
    if (config.hasPath("tours.searcher.v2.host")) {
      new Searcher2Adapter(config.getString("tours.searcher.v2.host"),
        config.getInt("tours.searcher.v2.port"), hotelsIndex)
    }
    else null
  }

  val throttler = new Throttler(
    config.getString("tours.hydra.limiter.host"),
    config.getInt("tours.hydra.limiter.port"),
    httpClient
  )

  val tourOperatorAvailability = new TourOperatorAvailability(
    config.getString("tours.hydra.limiter.host"),
    config.getInt("tours.hydra.limiter.port"),
    httpClient
  )

  val hotelProviderAvailability = new HotelProvidersAvailability(
    config.getString("tours.hydra.limiter.host"),
    config.getInt("tours.hydra.limiter.port"),
    httpClient
  )

  private val ostrovokFencedRatio = {
    val path = "tours.partners.ostrovok.fencedRatio"
    if (config.hasPath(path)) config.getDouble(path) else 0.0
  }

  val partner2config = PartnerConfigs.parse(config)

  lazy val enableClickHouseWrites = new SharedValue[Boolean](zkClient, "/clickhouse/searcher_wr_en", false, BooleanSerializer)

  lazy val routeesContext = new RouteesContext(departures, regionTree, None)

  lazy val cmHotelsBillingIndex = CMHotelsBillingIndex.from(hotelsIndex :: HNil, extDataService, extDataUpdateChecker)

  val hotelPartner2commonSearcher = partner2config.values
    .filter(_.useCommonClient)
    .filter(_.searchParameters.isDefined)
    .filter(_.searchParameters.get.searchType == SearchType.ROOMS)
    .filter(_.searchUrl.nonEmpty)
    .map { config =>
      val partner = config.partner
      val withCMBilling = Partners.isChannelManager(partner) && config.searchParameters.get.isBillingImportant
      val billingIndex = if (withCMBilling) Some(cmHotelsBillingIndex) else None
      val default = new CommonHotelProviderClient(hotelsIndex, pansionUnifier, geoMappingHolder, externalHttpClient,
        partner, config.searchUrl.get, config.headers, cmHotelBillingIndex = billingIndex)
      val wrapped = wrapSourceClient(default, partner)
      partner -> sourceSearcher(wrapped, hotelProviders, partner)
    }.toMap

  val tourPartner2commonSearcher = partner2config.values
    .filter(_.useCommonClient)
    .filter(_.searchParameters.isDefined)
    .filter(_.searchParameters.get.searchType == SearchType.TOURS)
    .filter(_.searchUrl.nonEmpty)
    .filter(_.actualizeUrl.nonEmpty)
    .map { config =>
      val partner = config.partner
      val defaultParser = new CommonTourOperatorResponseParser(hotelsIndex, pansionUnifier,
        geoMappingHolder, airports, partner, regionTree)
      val defaultClient = new CommonTourOperatorClient(
        searchUrl = config.searchUrl.get,
        actualizeUrl = config.actualizeUrl.get,
        geoMappingHolder, hotelsIndex, externalHttpClient, defaultParser,
        partner, config.searchParameters.get.explicitRegionType, regionTree
      )
      val wrappedClient = wrapSourceClient(defaultClient, partner)
      partner -> sourceSearcher(wrappedClient, tourOperators, partner)
    }.toMap

  val ltClient = {
    val actualizationParser = new ActualizationParser(geoMappingHolder)
    val baseClient = new DefaultLtClient(hotelsIndex, geoMappingHolder, regionTree, externalHttpClient,
      actualizationParser, tourOperators, pansionUnifier, partner2config(Partners.lt)) with InstrumentedLtClient
    new ThrottledLtClient(baseClient, throttler)
  }
  val ltSearcher = akkaSystem.actorOf(
    Props(new LevelTravelSearcher(ltClient, hotelsIndex, geoMappingHolder, regionTree)),
    "lt-searcher"
  )

  val hotelsCombinedSearcher = {
    val id2key = HotelsCombinedId2Key.from(extDataService, extDataUpdateChecker)
    val client = new DefaultHotelsCombinedClient(HotelsCombinedHttp.partner,
      hotelsIndex, externalHttpClient, hotelProviders
    ) with InstrumentedHotelsCombinedClient
    val throttledClient = new ThrottledHotelsCombinedClient(client, throttler)
    akkaSystem.actorOf(
      Props(new HotelsCombinedSearcher(client, hotelsIndex, geoMappingHolder, id2key,
        partner2config(Partners.hotelsCombined))),
      "hotelscombined-searcher"
    )
  }

  private val gzipFeature = new GZIPFeature
  gzipFeature.setThreshold(1024 * 1024)
  private val loggingFeature = new LoggingFeature(1024 * 1024, true, false)

  val sunmarClient: SunmarClient = {

    val searchService = new SearchService().getBasicHttpEndpoint(gzipFeature, loggingFeature)
    for ((host, port) <- httpProxy) WsdlUtils.injectProxy(searchService, host, port)
    WsdlUtils.injectTimeouts(searchService, 5.seconds, 30.seconds)

    val cred = new OTICredential
    cred.setUsername("Yandex")
    cred.setPassword("revoked") // todo: password removed from code. Need to get a new one and use secret storage when sumar is used again

    val baseClient = new DefaultSunmarClient(searchService, cred) with InstrumentedSunmarClient
    new ThrottledSunmarClient(baseClient, throttler)
  }
  val sunmarSearcher = {
    val translator = new SunmarTranslator(geoMappingHolder, regionTree, hotelsIndex, pansionUnifier)
    akkaSystem.actorOf(Props(new SunmarSearcher(sunmarClient, translator)), "sunmar-searcher")
  }

  val oktogoClient = {
    val service = new TravelApiService().getTravelApiServiceSoap(gzipFeature, loggingFeature)
    for ((host, port) <- httpProxy) WsdlUtils.injectProxy(service, host, port)
    WsdlUtils.injectTimeouts(service, 5.seconds, 60.seconds)
    val translator = new OktogoTranslator(geoMappingHolder, hotelsIndex, pansionUnifier)
    val default = new DefaultOktogoClient(service, hotelsIndex, translator)
    wrapSourceClient(default, Partners.oktogo)
  }

  val ostrovokv3Client = {
    val responseParser = new OstrovokV3ResponseParser(hotelsIndex, pansionUnifier)
    val default = new DefaultOstrovokV3Client(hotelsIndex, geoMappingHolder, regionTree, externalHttpClient,
      responseParser, partner2config(Partners.ostrovokv3), ostrovokFencedRatio, partner2config(Partners.ostrovokFenced),
      akkaSystem.eventStream)
    wrapSourceClient(default, Partners.ostrovokv3)
  }

  val bookingClient = {
    val responseParser = new BookingResponseParser(hotelsIndex, pansionUnifier)
    val default = new DefaultBookingClient(hotelsIndex, geoMappingHolder, regionTree, externalHttpClient,
      responseParser, partner2config(Partners.booking))
    wrapSourceClient(default, Partners.booking)
  }

  val oktogoSearcher = sourceSearcher(oktogoClient, hotelProviders, Partners.oktogo)
  val bookingSearcher = sourceSearcher(bookingClient, hotelProviders, Partners.booking)
  val ostrovokv3Searcher = sourceSearcher(ostrovokv3Client, hotelProviders, Partners.ostrovokv3)

  val hotelSearcherActor = akkaSystem.actorOf(Props(new SearcherActor(
    hotelsStorage,
    Map(
      Partners.ostrovokv3 -> ostrovokv3Searcher,
      Partners.booking -> bookingSearcher,
      Partners.hotelsCombined -> hotelsCombinedSearcher
//      , Partners.oktogo -> oktogoSearcher
    ) ++ hotelPartner2commonSearcher,
    hotelProviders,
    hotelsIndex,
    regionTree,
    hotelProviderAvailability
  )), "hotel-searcher-actor")

  val searcherActor = akkaSystem.actorOf(Props(new SearcherActor(
    toursStorage,
    Map(
      Partners.lt -> ltSearcher,
      Partners.sunmar -> sunmarSearcher
    ) ++ tourPartner2commonSearcher,
    tourOperators,
    hotelsIndex,
    regionTree,
    tourOperatorAvailability
  )), "tour-searcher-actor")

  val roomsSearcher = new LocalHotelSearchService(
    hotelsStorage,
    hotelsLongCache,
    hotelSearcherActor,
    geoMappingHolder,
    SearchType.ROOMS,
    regionTree,
    hotelsIndex,
    searchSettings,
    searcher2adapter
  )

  val offersSearcher = new LocalOffersSearchService(
    toursStorage,
    toursLongCache,
    searcherActor,
    geoMappingHolder,
    regionTree,
    hotelsIndex,
    searchSettings,
    searcher2adapter
  )

  val minPriceService = new LocalMinPriceService(minPriceStorage, directionPriceDao)
  val directionInterestService = new DummyDirectionInterestService
  val recommendService = {
    val directionRecommendationService =
      new DirectionRecommendationService(directionInterestService, directionPriceStorage,
        hotelsDirectionPriceStorage, departures, SearchType.TOURS)

    new LocalRecommendService(
      directionRecommendationService,
      directions,
      directionsStats,
      directionPriceStorage,
      regionTree,
      SearchType.TOURS
    )
  }

  val roomsMinPriceService = new LocalMinPriceService(hotelsMinPriceStorage, hotelsDirectionPriceDao)
  val roomsRecommendService = {
    val directionRecommendationService =
      new DirectionRecommendationService(directionInterestService, directionPriceStorage,
        hotelsDirectionPriceStorage, departures, SearchType.ROOMS)

    new LocalRecommendService(
      directionRecommendationService,
      directions,
      directionsStats,
      hotelsDirectionPriceStorage,
      regionTree,
      SearchType.ROOMS
    )
  }

  val tourHandler = akkaSystem.actorOf(Props(
    new SearcherHandler(offersSearcher, minPriceService, recommendService, routeesContext.searchRequest)
  ), "tours-handler")
  val roomsHandler = akkaSystem.actorOf(Props(
    new SearcherHandler(roomsSearcher, roomsMinPriceService, roomsRecommendService, routeesContext.roomSearchRequest)
  ), "hotels-handler")

  val rootHandler = Props(new RootSearcherHttpHandler(
    routeesContext,
    tourHandler,
    roomsHandler,
    flightsSearchService,
    transferSearchService,
    enableClickHouseWrites
  ))

  val pingControl = {
    val pinger = new PingControl
    pinger.scheduleRemoteCheck(config.getConfig("calendar"), httpClient, 3.seconds, 500.millis)(akkaSystem)
    pinger
  }

  val priceHistoryStorage = new PriceHistoryStorage(clickHouseClient, hotelsIndex, regionTree,
    geoMappingHolder, enableClickHouseWrites)(akkaSystem.dispatcher)

  onStart {
    log.info("Current clickhouse writing status: " + (if (enableClickHouseWrites.get) "Enabled" else "Disabled"))
    enableClickHouseWrites.onUpdate { enabled =>
      log.info("Updated clickhouse writing status: " + (if (enabled) "Enabled" else "Disabled"))
    }
    akkaSystem.actorOf(Props[HydraLogger], "hydra-logger")
    akkaSystem.actorOf(Props(new PriceHistoryTransfer(priceHistoryStorage)), "price-transfer")

    geoMappingHolder
    regionTree
    hotelsIndex
    directions
    departures

    startServer(rootHandler, config.getInt("component.port"), pingControl)
  }

  onStop {
    enableClickHouseWrites.close()
  }

  private def sourceSearcher[S <: Source : ClassTag](client: SourceClient[S],
                                                     sources: SearchSources[S],
                                                     partner: Partner) = {
    val sp = partner2config(partner).searchParameters.getOrElse(sys.error(s"Search parameter not defined for $partner"))
    akkaSystem.actorOf(Props(new SingleSourceSearcher(partner,
      client,
      sources,
      geoMappingHolder,
      hotelsIndex,
      billingService,
      regionTree,
      resortPriorities,
      searchSettings,
      sp.searchBySpan,
      sp.isBillingImportant && !Partners.isChannelManager(partner),
      sp.countrySearch,
      sp.isDepartureImportant,
      sp.useSmartFallback,
      sp.useSearchByHotelIds)))
  }

  private def wrapSourceClient[S <: Source](client: SourceClient[S], partner: Partner) = {
    val instrumented = new InstrumentedSourceClient[S](partner, client)
    val throttled = new ThrottledSourceClient[S](partner, instrumented, throttler)
    new FailLogSourceClient[S](throttled)
  }
}
