package ru.yandex.tours.backend.events

import akka.actor.Actor
import org.joda.time.{DateTime, DateTimeZone}
import org.slf4j.LoggerFactory
import ru.yandex.tours.events.SearchEvent
import ru.yandex.tours.events.SearchEvent._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 27.02.15
 */
class HydraLogger extends Actor {

  private val searchLogger = LoggerFactory.getLogger("tours.search.logs")
  private val format = "tours-searcher-log"
  private val project = "tours"
  private val locale = "ru"
  private val formatVersion = "1"

  private val prefix = s"tskv\ttskv_format=$format\tformat_version=$formatVersion\tlocale=$locale\tproject=$project"

  context.system.eventStream.subscribe(self, classOf[SearchEvent])

  private def logTSKV(component: String, data: Map[String, Any]) = {
    val ts = DateTime.now(DateTimeZone.UTC)
    val sb = new StringBuilder(prefix)
    sb.append("\tcomponent=").append(component)
    sb.append("\ttimestamp=").append(ts)
    for ((key, value) <- data) {
      sb ++= "\t"
      sb ++= key
      sb ++= "="
      sb ++= value.toString
    }
    searchLogger.info(sb.toString())
  }

  override def receive: Receive = {
    case e @ FoundSnippets(_, req, res) =>
      logTSKV(SearchEvent.SNIPPETS_SEARCH, e.asMap)
    case e @ FoundOffers(_, req, res) =>
      logTSKV(SearchEvent.OFFERS_SEARCH, e.asMap)
    case e @ FoundFlights(_, req, res) =>
      logTSKV(SearchEvent.FLIGHTS_SEARCH, e.asMap)
    case e @ FoundTransfers(_, req, res) =>
      logTSKV(SearchEvent.TRANSFER_SEARCH, e.asMap)
    case e @ Actualized(_, req, offer) =>
      logTSKV(SearchEvent.TOUR_ACTUALIZATION, e.asMap)
    case e @ OstrovokResult(_, req, _, res) =>
      logTSKV(e.component, e.asMap)
  }
}
