package ru.yandex.tours.backend.flight

import akka.actor.{Actor, Props, Status}
import akka.pattern._
import org.joda.time.DateTime
import ru.yandex.tours.avia.{Airports, AviaClient, FlightUtils}
import ru.yandex.tours.model.search.FlightSearchRequest
import ru.yandex.tours.model.search.SearchProducts.FlightTrip
import ru.yandex.tours.util.{Logging, Metrics}

import scala.collection.JavaConverters._
import scala.concurrent.Promise
import scala.concurrent.duration._
import scala.util.Try

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 09.02.16
 */
class AviaSearcher(aviaClient: AviaClient,
                   airports: Airports,
                   request: FlightSearchRequest,
                   promise: Promise[Seq[FlightTrip]]) extends Actor with Logging {

  import context.dispatcher

  protected val selfName = getClass.getSimpleName + " (" + request.sessionId + ")"

  private val metrics = Metrics("avia.searcher")
  private val timer = metrics.getTimer("time").time()
  private val errors = metrics.getMeter("errors")
  private val found = metrics.getMeter("found")
  private val notFound = metrics.getMeter("not_found")
  private val allInsane = metrics.getMeter("all_insane")

  private val knownCurrency = Set("RUR", "рубли")
  private val pollInterval = 3.seconds
  private case object DoPollRequest

  private var updating = false

  override def receive: Receive = {
    case AviaClient.Response(minPrices, None) if updating ⇒
      failure(new IllegalStateException("Got AviaClient.Response without search_id after request with update"))
    case res @ AviaClient.Response(minPrices, None) ⇒
      if (res.isUseful && minPrices.nonEmpty) {
        complete(minPrices)
      } else {
        updating = true
        aviaClient.getMinPrice(request, update = true).pipeTo(self)
      }
    case AviaClient.Response(minPrices, Some(searchId)) ⇒
      self ! DoPollRequest
      context.become(polling(searchId))
  }

  def polling(searchId: String): Receive = {
    case DoPollRequest =>
      aviaClient.getMinPriceUpdated(searchId).pipeTo(self)
    case AviaClient.ResponseUpdated(_, "querying") =>
      context.system.scheduler.scheduleOnce(pollInterval, self, DoPollRequest)
    case AviaClient.ResponseUpdated(Some(minPrices), "done") =>
      complete(minPrices)
    case AviaClient.ResponseUpdated(minPricesOpt, status) =>
      failure(new RuntimeException(s"Unexpected response from avia update: $minPricesOpt with status $status"))
  }

  override def unhandled(message: Any): Unit = message match {
    case Status.Failure(t) => failure(t)
    case _ => super.unhandled(message)
  }

  private def complete(minPrices: AviaClient.MinPrices): Unit = {
    val response = convertResponse(minPrices)
    val (valid, insaneFlights) = response.partition(isValidFlightTrip)
    collectMetrics(valid, insaneFlights)
    timer.stop()
    log.info(s"Found ${valid.size} flights for request $request")
    promise.complete(Try(valid))
    context.stop(self)
  }

  private def failure(t: Throwable): Unit = {
    promise.failure(t)
    errors.mark()
    log.error("Avia search failed with exception", t)
    context.stop(self)
  }

  private def collectMetrics(valid: Seq[FlightTrip], insaneFlights: Seq[FlightTrip]): Unit = {
    if (valid.nonEmpty) found.mark() else notFound.mark()
    if (insaneFlights.nonEmpty) {
      if (valid.isEmpty) allInsane.mark()
      log.warn(s"Found ${insaneFlights.size} insane flights for request $request")
    }
  }

  private def convertResponse(response: AviaClient.MinPrices): Seq[FlightTrip] = {
    val result = Vector.newBuilder[FlightTrip]
    val flights = response.flights.map(f => f.key -> f).toMap
    val companies = response.companies.map(c => c.id -> c).toMap

    def addFlight(minPrice: AviaClient.MinPrice): Unit = {
      if (knownCurrency.contains(minPrice.tariff.currency)) {
        for (variant <- minPrice.variants if variant.isUseful) {
          val builder = FlightTrip.newBuilder()
          builder.setMinPrice(minPrice.tariff.price)

          for (forward <- variant.forward) fillFlight(flights(forward), companies, builder.addForwardBuilder())
          for (backward <- variant.backward) fillFlight(flights(backward), companies, builder.addBackwardBuilder())
          result += builder.build()
        }
      } else {
        log.warn(s"Unexpected currency: ${minPrice.tariff.currency} in $minPrice for $request")
      }
    }

    for (direct <- response.direct) addFlight(direct)
    for (indirect <- response.indirect) addFlight(indirect)

    result.result()
  }

  private def isValidFlightTrip(flightTrip: FlightTrip): Boolean = {
    def check(flights: Seq[FlightTrip.Flight]): Boolean = {
      for (flight <- flights) {
        if (FlightUtils.getFlightDuration(flight, flight).getStandardMinutes <= 0) {
          log.warn(s"Negative flight duration for $request \n $flight")
          return false
        }
        if (airports.byAviaId(flight.getDeparture).isEmpty) {
          log.warn(s"Unknown departure airport id for $request: ${flight.getDeparture}")
          return false
        }
        if (airports.byAviaId(flight.getArrival).isEmpty) {
          log.warn(s"Unknown arrival airport id for $request: ${flight.getArrival}")
          return false
        }
      }
      for (Seq(first, second) <- flights.sliding(2)) {
        if (FlightUtils.getTransitionDuration(first, second).getStandardMinutes <= 0) {
          log.warn(s"Negative transition time for $request \n $first \n-----\n $second")
          return false
        }
      }
      true
    }

    check(flightTrip.getForwardList.asScala) && check(flightTrip.getBackwardList.asScala)
  }

  private def fillFlight(flight: AviaClient.Flight, companies: Map[Int, AviaClient.Company],
                         builder: FlightTrip.Flight.Builder): Unit = {
    builder
      .setFlightNumber(flight.number)
      .setDeparture(flight.departure_key)
      .setArrival(flight.arrival_key)
      .setDepartureTime(convertTime(flight.departure_key, flight.departure_datetime))
      .setArrivalTime(convertTime(flight.arrival_key, flight.arrival_datetime))
      .setDuration(flight.path_duration)
      .setCompany(companies(flight.company_id).title)
  }

  private def convertTime(airportId: String, dateTime: DateTime): String = dateTime.toString


  @scala.throws[Exception](classOf[Exception])
  override def preStart(): Unit = {
    log.info(s"Starting $selfName for request $request")
    aviaClient.getMinPrice(request, update = false).pipeTo(self)
  }

  override def postRestart(reason: Throwable): Unit = {
    super.postRestart(reason)
    log.error(s"Restarting AviaSearcher, this should not happen", reason)
  }

  override def postStop(): Unit = {
    log.info(s"$selfName finished for request $request")
  }
}

object AviaSearcher {
  def props(aviaClient: AviaClient,
            airports: Airports,
            request: FlightSearchRequest,
            promise: Promise[Seq[FlightTrip]]): Props = {
    Props(new AviaSearcher(aviaClient, airports, request, promise))
  }
}
