package ru.yandex.tours.backend.flight

import akka.actor.{ActorRef, ActorSystem}
import org.joda.time.{DateTime, LocalDate}
import ru.yandex.tours.events.SearchEvent.FoundFlights
import ru.yandex.tours.model.Prices.FlightMinPriceMatrix
import ru.yandex.tours.model.search.FlightSearchRequest
import ru.yandex.tours.model.search.SearchProducts.FlightTrip
import ru.yandex.tours.model.search.SearchResults.{ErrorCode, FlightSearchResult, ResultInfo, SearchProgress}
import ru.yandex.tours.search.settings.SearchSettingsHolder
import ru.yandex.tours.services.FlightSearchService
import ru.yandex.tours.storage.flight.FlightsDao
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.collections.SimpleBitSet

import scala.collection.JavaConverters._
import scala.concurrent.{ExecutionContext, Future, Promise}
import scala.util.{Failure, Success}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 09.02.16
 */
class LocalFlightSearchService(flightsStorage: FlightsDao,
                               searchSettingsHolder: SearchSettingsHolder,
                               rootSearcherActor: ActorRef,
                               simpleAviaSearcher: Option[SimpleAviaSearcher])
                              (implicit ec: ExecutionContext, akkaSystem: ActorSystem)
  extends FlightSearchService with Logging {

  override def searchFlights(request: FlightSearchRequest, canStartRequest: Boolean): Future[FlightSearchResult] = {
    if (searchSettingsHolder.isTooClose(request)) {
      Future.successful {
        flights(System.currentTimeMillis(), Seq.empty, successProgress, tooClose)
      }
    } else if (request.hotelRequest.when.isBefore(LocalDate.now())) {
      Future.successful {
        flights(System.currentTimeMillis(), Seq.empty, successProgress, notStarted)
      }
    } else if (request.airportId == "g225") {
      Future.successful {
        flights(System.currentTimeMillis(), Seq.empty, successProgress, tooLarge)
      }
    } else if (simpleAviaSearcher.isDefined ) {
      val started = System.currentTimeMillis()
      val result = for (res <- simpleAviaSearcher.get.getFlights(request))
        yield flights(started, res, successProgress)
      result.onComplete {
        case Success(res) =>
          akkaSystem.eventStream.publish(FoundFlights(DateTime.now, request, res))
          flightsStorage.saveFlights(request, res)
        case Failure(t) =>
          log.error(s"Failed to search flights. Request = $request", t)
      }
      result
    }
    else {
      flightsStorage.getFlights(request).flatMap {
        case Some(result) => Future.successful(result)
        case None if canStartRequest => startNewRequest(request)
        case None =>
          Future.successful {
            flights(System.currentTimeMillis(), Seq.empty, successProgress, notStarted)
          }
      }
    }
  }

  override def getFlightMatrix(request: FlightSearchRequest,
                               from: Seq[Int], airports: Seq[String]): Future[FlightMinPriceMatrix] = {

    val requests1 = for {
      airport <- airports
      if airport.nonEmpty
    } yield request.copy(airportId = airport)

    val requests2 = for {
      from <- from
    } yield request.withFrom(from)

    val requests = (requests1 ++ requests2).distinct

    val results = requests.map(req => flightsStorage.getFlights(req).map(req -> _))

    Future.fold(results)(FlightMinPriceMatrix.newBuilder()) {
      case (builder, (req, resOpt)) =>
        for {
          result <- resOpt
          if result.getFlightTripsCount > 0
        } {
          val minPrice = result.getFlightTripsList.asScala.map(_.getMinPrice).min
          builder.addPriceEntityBuilder()
            .setFrom(req.hotelRequest.from)
            .setAirportId(req.airportId)
            .setMinPrice(minPrice)
        }
        builder
    }.map(_.build())
  }

  private def startNewRequest(request: FlightSearchRequest): Future[FlightSearchResult] = {
    val started = System.currentTimeMillis()
    val result = flights(started, Seq.empty, emptyProgress)
    val promise = Promise.apply[Seq[FlightTrip]]()

    promise.future.onComplete {
      case Success(trips) =>
        val res = flights(started, trips, successProgress)
        akkaSystem.eventStream.publish(FoundFlights(DateTime.now, request, res))
        flightsStorage.saveFlights(request, res)
      case Failure(t) =>
        log.error(s"Failed to search flights. Request = $request", t)
        val res = flights(started, Seq.empty, failedProgress, resultInfo = allFailed)
        flightsStorage.saveFlights(request, res)
    }

    flightsStorage.saveFlights(request, result).map { _ =>
      rootSearcherActor ! RootAviaSearcherActor.SpawnSearcher(request, promise)
      result
    }
  }

  private def flights(started: Long, trips: Seq[FlightTrip], progress: SearchProgress,
                      resultInfo: ResultInfo = defaultResultInfo): FlightSearchResult =
    FlightSearchResult.newBuilder()
      .setCreated(started)
      .setUpdated(System.currentTimeMillis())
      .setProgress(progress)
      .setResultInfo(resultInfo)
      .addAllFlightTrips(trips.asJava)
      .build()

  private val emptyProgress = progress(0, 0, 1)
  private val successProgress = progress(1, 0, 1)
  private val failedProgress = progress(0, 1, 1)

  private def progress(success: Int, failure: Int, total: Int) = SearchProgress.newBuilder()
    .setIsFinished(success + failure >= total)
    .setOperatorTotalCount(total)
    .setOperatorCompleteCount(success + failure)
    .setOperatorFailedCount(failure)
    .setOperatorSkippedCount(0)
    .setOperatorSkippedSet(SimpleBitSet(Set.empty).packed)
    .build()

  private val defaultResultInfo = ResultInfo.newBuilder()
    .setIsFromLongCache(false)
    .build()

  private val notStarted = ResultInfo.newBuilder()
    .setIsFromLongCache(false)
    .setError(ErrorCode.NOT_STARTED)
    .build()

  private val tooLarge = ResultInfo.newBuilder()
    .setIsFromLongCache(false)
    .setError(ErrorCode.TOO_LARGE_DESTINATION)
    .build()

  private val tooClose = ResultInfo.newBuilder()
    .setIsFromLongCache(false)
    .setError(ErrorCode.TOO_CLOSE_DESTINATION)
    .build()

  private val allFailed = ResultInfo.newBuilder()
    .setIsFromLongCache(false)
    .setError(ErrorCode.ALL_SOURCES_FAILED)
    .build()
}
